/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.content.archive.ArchiveConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.UsersQuery;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for ODF right
 */
public class ODFRightHelper extends AbstractLogEnabled implements Serviceable, Component, Contextualizable, Initializable
{
    /** The avalon role */
    public static final String ROLE = ODFRightHelper.class.getName();
    
    /** Request attribute name for storing current educational paths */
    public static final String REQUEST_ATTR_EDUCATIONAL_PATHS = ODFRightHelper.class.getName() + "$educationalPath";
    
    /** Attribute path for contributor role */
    public static final String CONTRIBUTORS_FIELD_PATH = "odf-contributors";
    /** Attribute path for manager role */
    public static final String MANAGERS_FIELD_PATH = "odf-managers";
    
    private static final String __PARENTS_CACHE_ID = ODFRightHelper.class.getName() + "$parentsCache";
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    /** The avalon context */
    protected Context _context;
    /** Ametys Object Resolver */
    protected AmetysObjectResolver _resolver;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__PARENTS_CACHE_ID))
        {
            _cacheManager.createRequestCache(__PARENTS_CACHE_ID,
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_RIGHTS_PARENT_ELEMENTS_LABEL"),
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_RIGHTS_PARENT_ELEMENTS_DESCRIPTION"),
                false
            );
        }
    }
    
    /**
     * Get the id of profile for contributors
     * @return the id of profile for contributors
     */
    public String getContributorProfileId()
    {
        return Config.getInstance().getValue("odf.profile.contributor");
    }
    
    /**
     * Get the id of profile for managers
     * @return the id of profile for managers
     */
    public String getManagerProfileId()
    {
        return Config.getInstance().getValue("odf.profile.manager");
    }
    
    /**
     * Get the contributors of a {@link ProgramItem} or a {@link OrgUnit}
     * @param content the program item or the orgunit
     * @return the contributors or null if not found
     */
    public UserIdentity[] getContributors(Content content)
    {
        if (content instanceof OrgUnit || content instanceof ProgramItem)
        {
            return content.getValue(CONTRIBUTORS_FIELD_PATH);
        }
        
        return null;
    }
    
    /**
     * Build a user query on contributors field
     * @param users The users to test.
     * @return the user query
     */
    public Query getContributorsQuery(UserIdentity... users)
    {
        return new UsersQuery(CONTRIBUTORS_FIELD_PATH + "_s", Operator.EQ, users);
    }
    
    /**
     * Get the managers of a {@link ProgramItem} or a {@link OrgUnit}
     * @param content the program item or the orgunit
     * @return the managers or null if not found
     */
    public UserIdentity[] getManagers(Content content)
    {
        if (content instanceof OrgUnit || content instanceof ProgramItem)
        {
            return content.getValue(MANAGERS_FIELD_PATH);
        }
        
        return null;
    }
    
    /**
     * Build a user query on managers field
     * @param users The users to test.
     * @return the user query
     */
    public Query getManagersQuery(UserIdentity... users)
    {
        return new UsersQuery(MANAGERS_FIELD_PATH + "_s", Operator.EQ, users);
    }
    
    /**
     * Get the query to search for contents for which the user has a role
     * @param user the user
     * @return the query to filter on user permission
     */
    public Query getRolesQuery(UserIdentity user)
    {
        List<Query> userQueries = new ArrayList<>();
        
        userQueries.add(getContributorsQuery(user));
        userQueries.add(getManagersQuery(user));
        
        return new OrQuery(userQueries);
    }
    
    /**
     * Get the ODF contents with users with given role
     * @param roleAttributePath the attribute path for role
     * @return the contents with user(s) set with a role
     */
    public AmetysObjectIterable<Content> getContentsWithRole(String roleAttributePath)
    {
        Expression cTypeExpr = new ContentTypeExpression(org.ametys.plugins.repository.query.expression.Expression.Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE, SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, ContainerFactory.CONTAINER_CONTENT_TYPE, CourseFactory.COURSE_CONTENT_TYPE, OrgUnitFactory.ORGUNIT_CONTENT_TYPE);
        Expression roleExpr =  new RoleExpression(roleAttributePath);
        
        String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, roleExpr));
        try (AmetysObjectIterable<Content> contents = _resolver.query(query))
        {
            return contents;
        }
    }
    
    /**
     * Get the programs items the user is set with a role
     * @param user The user
     * @param roleAttributePath the attribute path for role
     * @return the program items the user is set with a role
     */
    public AmetysObjectIterable<Content> getContentsWithUserAsRole(UserIdentity user, String roleAttributePath)
    {
        Expression cTypeExpr = new ContentTypeExpression(org.ametys.plugins.repository.query.expression.Expression.Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE, SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, ContainerFactory.CONTAINER_CONTENT_TYPE, CourseFactory.COURSE_CONTENT_TYPE, OrgUnitFactory.ORGUNIT_CONTENT_TYPE);
        Expression roleExpr =  new RoleExpression(roleAttributePath, user);
        
        String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, roleExpr));
        try (AmetysObjectIterable<Content> contents = _resolver.query(query))
        {
            return contents;
        }
    }
    
    /**
     * Get the parents of a ODF content from a rights perspective
     * @param content the content
     * @param permissionCtx The permission context to compute rights. The permission context allows to restrict parents to parents which are part of a given ancestor. See #withAncestor method of {@link PermissionContext}
     * @return the parents of the content for rights computing
     */
    public Set<AmetysObject> getParents(Content content, PermissionContext permissionCtx)
    {
        Cache<CacheKey, Set<AmetysObject>> cache = _getParentsCache();
        
        CacheKey key = CacheKey.of(content.getId(), permissionCtx);
        return cache.get(key, id -> computeParents(content, permissionCtx));
    }
    /**
     * Compute the parents of a ODF content from a rights perspective
     * @param content the content
     * @param permissionCtx The permission context to compute rights.
     * @return the parents of the content for rights computing
     */
    protected Set<AmetysObject> computeParents(Content content, PermissionContext permissionCtx)
    {
        Set<AmetysObject> parents = new HashSet<>();
        
        if (content instanceof ProgramItem programItem)
        {
            if (permissionCtx.getAncestor() != null && programItem.equals(permissionCtx.getAncestor()))
            {
                // reset ancestor
                permissionCtx.withAncestor(null);
            }
            parents.addAll(computeParentProgramItem(programItem, permissionCtx));
            parents.addAll(computeOrgUnits(programItem, permissionCtx));
        }
        else if (content instanceof OrgUnit orgUnit)
        {
            OrgUnit parentOrgUnit = orgUnit.getParentOrgUnit();
            if (parentOrgUnit != null)
            {
                parents.add(parentOrgUnit);
            }
        }
        else if (content instanceof CoursePart coursePart)
        {
            List<Course> parentCourses = coursePart.getCourses();
            if (!parentCourses.isEmpty())
            {
                parents.addAll(parentCourses);
            }
        }
        
        // default
        AmetysObject parent = content.getParent();
        boolean parentAdded = false;
        if (parent instanceof AmetysObjectCollection collection && (RepositoryConstants.NAMESPACE_PREFIX + ":contents").equals(collection.getName()))
        {
            Request request = ContextHelper.getRequest(_context);
            String originalWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
            if (ArchiveConstants.ARCHIVE_WORKSPACE.equals(originalWorkspace))
            {
                try
                {
                    RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
                    AmetysObject parentFromDefault = _resolver.resolveByPath(parent.getPath());
                    parents.add(parentFromDefault);
                    parentAdded = true;
                }
                finally
                {
                    RequestAttributeWorkspaceSelector.setForcedWorkspace(request, originalWorkspace);
                }
            }
        }
        if (!parentAdded)
        {
            parents.add(parent);
        }
        
        return parents;
    }
    
    /**
     * Get the parents of a program item from a rights perspective
     * @param programItem the program item
     * @param permissionCtx The permission context to compute rights.
     * @return the parent program items for rights computing
     */
    protected List<ProgramItem> computeParentProgramItem(ProgramItem programItem, PermissionContext permissionCtx)
    {
        if (programItem instanceof Course course)
        {
            return computeParentProgramItem(course, permissionCtx);
        }
        else
        {
            return _odfHelper.getParentProgramItems(programItem, permissionCtx.getAncestor());
        }
    }
    
    /**
     * Get the parents of a course from a rights perspective
     * @param course the course
     * @param permissionCtx The permission context to compute rights.
     * @return the course parents for rights computing
     */
    protected List<ProgramItem> computeParentProgramItem(Course course, PermissionContext permissionCtx)
    {
        // Skip course lists to avoid unecessary rights computing on a course list
        List<ProgramItem> parents = new ArrayList<>();
        
        List<CourseList> parentCourseLists = course.getParentCourseLists();
        
        boolean ancestorIsParentCL = permissionCtx.getAncestor() != null && parentCourseLists.stream().anyMatch(cl -> cl.equals(permissionCtx.getAncestor()));
        if (ancestorIsParentCL)
        {
            // Required ancestor is a direct parent course list
            parents.addAll(_odfHelper.getParentProgramItems(permissionCtx.getAncestor(), null));
        }
        else
        {
            for (CourseList parentCL : parentCourseLists)
            {
                parents.addAll(_odfHelper.getParentProgramItems(parentCL, permissionCtx.getAncestor()));
            }
        }
        
        return parents;
    }
    
    /**
     * Get the orgunits of a program item
     * @param programItem the program item
     * @param permissionCtx The permission context to compute rights.
     * @return the orgunits
     */
    protected List<OrgUnit> computeOrgUnits(ProgramItem programItem, PermissionContext permissionCtx)
    {
        Set<String> ouIds = new HashSet<>();
        
        if (programItem instanceof CourseList courseList)
        {
            List<Course> parentCourses = courseList.getParentCourses();
            for (Course parentCourse : parentCourses)
            {
                ouIds.addAll(parentCourse.getOrgUnits());
            }
        }
        else
        {
            ouIds.addAll(programItem.getOrgUnits());
        }

        return ouIds.stream()
                    .filter(Objects::nonNull)
                    .map(_resolver::resolveById)
                    .map(OrgUnit.class::cast)
                    .toList();
    }
    
    private Cache<CacheKey, Set<AmetysObject>> _getParentsCache()
    {
        return _cacheManager.get(__PARENTS_CACHE_ID);
    }
    
    static class CacheKey extends AbstractCacheKey
    {
        CacheKey(String contentId, PermissionContext permissionCtx)
        {
            super(contentId, permissionCtx);
        }

        static CacheKey of(String contentId, PermissionContext permissionCtx)
        {
            return new CacheKey(contentId, permissionCtx);
        }
    }
    
    /**
     * Class representing the permission context for parents computation.
     * The permission context is composed by the initial content and an optional program part ancestor to restrict parents to parents which is part of this ancestor.
     */
    public static class PermissionContext
    {
        private Content _initialContent;
        private ProgramPart _ancestor;
        
        /**
         * Constructor
         * @param initialContent the initail content
         */
        public PermissionContext(Content initialContent)
        {
            this(initialContent, null);
        }
        
        /**
         * Constructor
         * @param initialContent the initail content
         * @param ancestor The ancestor. Can be null.
         */
        public PermissionContext(Content initialContent, ProgramPart ancestor)
        {
            _initialContent = initialContent;
            _ancestor = ancestor;
        }
        
        /**
         * Get the initial content
         * @return the initial content
         */
        public Content getInitialContent()
        {
            return _initialContent;
        }
        
        /**
         * Set an ancestor. Only parents that part of this ancestor will be returned.
         * @param ancestor the ancestor
         */
        public void withAncestor(ProgramPart ancestor)
        {
            _ancestor = ancestor;
        }
        
        /**
         * Get the ancestor
         * @return the ancestor. Can be null.
         */
        public ProgramPart getAncestor()
        {
            return _ancestor;
        }
    }
    
    /**
     * Class representing the permission context for parents computation for a contextualized content.
     * The permission context is composed by the initial content and a {@link EducationalPath}
     */
    public static class ContextualizedPermissionContext
    {
        private Content _initialContent;
        private EducationalPath _educationalPath;
        
        /**
         * Constructor
         * @param initialContent the initial content
         */
        public ContextualizedPermissionContext(Content initialContent)
        {
            this(initialContent, null);
        }
        
        /**
         * Constructor
         * @param initialContent the initial program item
         * @param educationalPath The educational path. Can be null.
         */
        public ContextualizedPermissionContext(Content initialContent, EducationalPath educationalPath)
        {
            _initialContent = initialContent;
            _educationalPath = educationalPath;
        }
        
        /**
         * Get the initial content
         * @return the initial content
         */
        public Content getInitialContent()
        {
            return _initialContent;
        }
        
        /**
         * Get the educational path in permission context
         * @return the educational path
         */
        public EducationalPath getEducationalPath()
        {
            return _educationalPath;
        }
        
        /**
         * Set an educational path. Only parent that part of this educational will be returned.
         * @param educationalPath the educationa path
         */
        public void withEducationalPath(EducationalPath educationalPath)
        {
            _educationalPath = educationalPath;
        }
    }
    
    /**
     * Record representing a contextualized content
     * @param content the content
     * @param path the context as a {@link EducationalPath}
     */
    public record ContextualizedContent(Content content, EducationalPath path) { }
    
    /**
     * Expression for ODF role expression
     *
     */
    public class RoleExpression implements Expression
    {
        private String _attributePath;
        private UserIdentity _user;

        /**
         * Constructor
         * @param attributePath the path of the role attribute
         */
        public RoleExpression(String attributePath)
        {
            this(attributePath, null);
        }
        
        /**
         * Constructor
         * @param attributePath the path of the role attribute
         * @param user the user. Can be null.
         */
        public RoleExpression(String attributePath, UserIdentity user)
        {
            _attributePath = attributePath;
            _user = user;
        }
        
        public String build()
        {
            StringBuilder buff = new StringBuilder();
            
            buff.append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append(_attributePath).append("/*/").append('@').append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append("login");
            if (_user != null)
            {
                buff.append(" " + Operator.EQ);
                buff.append(" '" + _user.getLogin() + "'");
            }
            
            buff.append(LogicalOperator.AND.toString());
            
            buff.append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append(_attributePath).append("/*/").append('@').append(RepositoryConstants.NAMESPACE_PREFIX).append(':').append("population");
            if (_user != null)
            {
                buff.append(" " + Operator.EQ);
                buff.append(" '" + _user.getPopulationId() + "'");
            }
            
            return buff.toString();
        }
    }
}
