/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation.skill;

import java.util.Map;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.odf.program.Program;

/**
 * Observer to store the skills of a program before modification.
 * When the modified content is a Program, keep its "transversalSkills" in the request to be able to compare them after modification and edit its courses
 */
public class UpdateProgramSkillStep1Observer extends AbstractSkillsStepObserver
{
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_MODIFYING;
    }
    
    @Override
    protected boolean supportsContent(Content content)
    {
        // Only support Programs
        return content instanceof Program;
    }
    
    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Program program = (Program) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        // When still modifying the content, save the transversal skills in the request in order to be able to compare them later
        ContentValue[] previousTransversalSkillsContentValue = program.getValue(Program.TRANSVERSAL_SKILLS, false, new ContentValue[0]);
        
        // Save the skills before modification in the request
        _setRequestAttribute("previousTransversalSkills", program.getId(), previousTransversalSkillsContentValue);
    }
}
