/*
 *  Copyright 2026 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program.properties;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.model.properties.AbstractIndexableContentProperty;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;

/**
 * Property on the computed period.
 */
public class ComputedPeriodProperty extends AbstractIndexableContentProperty<Content>
{
    private ODFHelper _odfHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    public String getContentTypeId()
    {
        return OdfReferenceTableHelper.PERIOD;
    }
    
    @Override
    protected Set<? extends ModifiableContent> _getLinkedContents(Content content)
    {
        Set<ModifiableContent> periods = _getPeriods(content)
                .distinct()
                .collect(Collectors.toSet());
        
        // Do not return periods if there are multiple
        return periods.size() > 1 ? Set.of() : periods;
    }
    
    private Stream<ModifiableContent> _getPeriods(Content content)
    {
        return Optional.of(content)
            .filter(c -> c.hasValue("period"))
            .map(c -> c.<ContentValue>getValue("period"))
            .flatMap(ContentValue::getContentIfExists)
            .map(Stream::of)
            // If not found, get periods from parents
            .orElseGet(() -> _getPeriodsFromParents((ProgramItem) content));
    }
    
    private Stream<ModifiableContent> _getPeriodsFromParents(ProgramItem programItem)
    {
        return _odfHelper.getParentProgramItems(programItem)
            .stream()
            .map(Content.class::cast)
            .flatMap(this::_getPeriods);
    }
}
