/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.util.List;
import java.util.Map;

import org.ametys.cms.repository.Content;
import org.ametys.core.ui.ClientSideElement.Script;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.Prioritizable;

/**
 * Interface for an ODF tree indicator
 *
 */
public interface ODFTreeIndicator extends Prioritizable
{
    /**
     * Get the id
     * @return the id
     */
    public String getId();
    
    /**
     * Get the indicator label
     * @return the label
     */
    public I18nizableText getLabel();
    
    /**
     * Get the indicator description
     * @return the description
     */
    public I18nizableText getDescription();
    
    /**
     * The indicator glyph icon
     * @return the CSS class for glyph icon
     */
    public String getIconGlyph();
    
    /**
     * Get scripts that will be used on client side.
     * @param contextParameters Contextuals parameters transmitted by the environment.
     * @return The list of scripts or an empty list.
     */
    public List<Script> getScripts(Map<String, Object> contextParameters);
    
    /**
     * Get the JS function to determines if indicator is available for a content. Can return <code>null</code> to use default one.
     * @return the JS function or <code>null</code> to use default matcher
     */
    public String getMatchJSFunction();
    
    /**
     * Get the JS function to display the indicator. Can return <code>null</code> to use default one.
     * @return the JS function or <code>null</code> to use default applyier
     */
    public String getApplyJSFunction();
    
    /**
     * Get the indicator values for a program item. Return <code>null</code> if the indicator is not available for this program item
     * @param content the content
     * @return the indicator data for this program item. Can be null.
     */
    public IndicatorData getIndicatorData(Content content);
    
    /**
     * Record for indicator data
     * @param tooltip The tooltip
     * @param text The text value. Can be null
     * @param cssClass the CSS class for the icon glyph. Can be null
     * @param additionalData the additional data. Can be null or empty.
     */
    public record IndicatorData(I18nizableText tooltip, I18nizableText text, String cssClass, Map<String, Object> additionalData) { /* empty */ }
        
} 
