/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Helper class for ODF contents.
 * @private
 */
Ext.define('Ametys.odf.helper.Content', {
    singleton: true,
    
    /**
     * Show the list of invalidated contents 
     * @param {Object/Object[]} contents Content of an array of contents with the invalidated referenced contents as following:
     * @param {String} contents.id The id of the root content which references invalidated contents
     * @param {String} contents.title (required) The title of the root content which references invalidated contents
     * @param {Object[]} contents.invalidatedContents (required) The invalidated contents with id, code and title
     * @param {String} [dialogTitle] The dialog title
     * @param {String} [dialogStartHintMsg] The start hint message
     * @param {String} [dialogEndHintMsg] The end hint message
     * @param {String} [contentIntroHintMsg] The introduction message for each contents
     * @param {String} [dialogIcon=Ext.Msg.INFO] The path to icon
     * @param {Boolean} [withConfirm=false] true to display confirm dialog box
     * @param {Function} [confirmCb] Function to call on confirm. Required if withConfirm is set to true
     */
    showInvalidatedContents: function(contents, dialogTitle, dialogStartHintMsg, dialogEndHintMsg, contentIntroHintMsg, dialogIcon, withConfirm, confirmCb)
    {
        contents = Ext.Array.from(contents);
        
        var infoMsg = dialogStartHintMsg || '';
        
        Ext.Array.each(contents, function(content) {
            contentIntroHintMsg = contentIntroHintMsg || "{{i18n PLUGINS_ODF_HELPER_SHOW_INVALIDATED_CONTENTS_DIALOG_HINT1}}";
            
            infoMsg += Ext.String.format(contentIntroHintMsg, content.title);
            infoMsg += '<ul>';
            
            Ext.Array.each(content.invalidatedContents, function(invalidatedContent) {
                var href = "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-content', {id:'" + invalidatedContent.id + "'});})()";
                infoMsg += '<li><a href="' + href + '">' + Ext.String.escapeHtml(invalidatedContent.title) + (invalidatedContent.code ? ' (' + invalidatedContent.code + ')' : '') + '</a></li>';
            });
        
            infoMsg += '</ul>';
        })
        
        infoMsg += dialogEndHintMsg || "{{i18n PLUGINS_ODF_HELPER_SHOW_INVALIDATED_CONTENTS_DIALOG_HINT2}}";
        
        Ametys.Msg.show({
            title: dialogTitle || "{{i18n PLUGINS_ODF_HELPER_SHOW_INVALIDATED_CONTENTS_DIALOG_TITLE}}",
            msg: infoMsg,
            maxHeight: 500,
            buttons: withConfirm ? Ext.Msg.YESNO : Ext.Msg.OK,
            icon: dialogIcon || Ext.Msg.INFO,
            fn: withConfirm ? function(btn) { if (btn == 'yes') { confirmCb() }} : null
        });
        
    }
});
    