/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.lheo;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.component.Component;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.google.common.base.Splitter;

/**
 * Class utils for LHEO XML export
 */
public class LHEOUtils extends AbstractLogEnabled implements Component
{
    /** The Avalon role */
    public static final String ROLE = LHEOUtils.class.getName();
    
    /**
     * Create some LHEO elements for tag &gt;coordonnees&lt;
     * <br>Contains the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param civility the civility. Can be null.
     * @param lastname the lastname. Can be null.
     * @param firstname the firstname. Can be null.
     * @param address the address. Can be null.
     * @throws SAXException if a saxing exception occurred
     */
    public void createCoordinateLHEOElementsPart1(ContentHandler contentHandler, Content content, String civility, String lastname, String firstname, String address) throws SAXException
    {
        List<String> lines = StringUtils.isNotBlank(address) ? Splitter.fixedLength(50).splitToList(address) : new ArrayList<>();
        createCoordinateLHEOElementsPart1(contentHandler, content, civility, lastname, firstname, lines);
    }
    
    /**
     * Create some LHEO elements for tag &gt;coordonnees&lt;
     * <br>Contains the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param civility the civility. Can be null.
     * @param lastname the lastname. Can be null.
     * @param firstname the firstname. Can be null.
     * @param lines the address line. Can be empty.
     * @throws SAXException if a saxing exception occurred
     */
    public void createCoordinateLHEOElementsPart1(ContentHandler contentHandler, Content content, String civility, String lastname, String firstname, List<String> lines) throws SAXException
    {
        // <civilite>
        createLHEOElement(contentHandler, content, "civilite", civility, 1, 50);
        
        // <nom>
        createLHEOElement(contentHandler, content, "nom", lastname, 1, 50);
        
        // <prenom>
        createLHEOElement(contentHandler, content, "prenom", firstname, 1, 50);
        
        // <ligne>
        int size = lines.size();
        if (size > 3)
        {
            size = 3;
            getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value " + StringUtils.join(lines, "") + " can be set in 3 lines. It will be truncated.");
        }
        
        for (int i = 0; i < size; i++)
        {
            createLHEOElement(contentHandler, content, "ligne", lines.get(i), 1, 50);
        }
    }
    
    /**
     * Create some LHEO elements for tag &gt;coordonnees&lt;
     * <br>Contains the following XML tags:
     * <br>[0,1] &lt;telfixe&gt;
     * <br>[0,1] &lt;portable&gt;
     * <br>[0,1] &lt;fax&gt;
     * <br>[0,1] &lt;courriel&gt;
     * <br>[0,1] &lt;web&gt;
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param telFix the telFix. Can be null.
     * @param portable the portable. Can be null.
     * @param fax the fax. Can be null.
     * @param mail the mail. Can be null.
     * @param urlWeb the url web. Can be null.
     * @throws SAXException if a saxing exception occurred
     */
    public void createCoordinateLHEOElementsPart2(ContentHandler contentHandler, Content content, String telFix, String portable, String fax, String mail, String urlWeb) throws SAXException
    {
        // <telfixe>
        if (StringUtils.isNotBlank(telFix))
        {
            XMLUtils.startElement(contentHandler, "telfixe");
            createLHEOElement(contentHandler, content, "numtel", telFix, 1, 25);
            XMLUtils.endElement(contentHandler, "telfixe");
        }
        
        // <portable>
        if (StringUtils.isNotBlank(portable))
        {
            XMLUtils.startElement(contentHandler, "portable");
            createLHEOElement(contentHandler, content, "numtel", portable, 1, 25);
            XMLUtils.endElement(contentHandler, "portable");
        }
        
        // <fax>
        if (StringUtils.isNotBlank(fax))
        {
            XMLUtils.startElement(contentHandler, "fax");
            createLHEOElement(contentHandler, content, "numtel", fax, 1, 25);
            XMLUtils.endElement(contentHandler, "fax");
        }
        
        // <courriel>
        createLHEOElement(contentHandler, content, "courriel", mail, 3, 160);
        
        // <web>
        if (StringUtils.isNotBlank(urlWeb))
        {
            XMLUtils.startElement(contentHandler, "web");
            createLHEOElement(contentHandler, content, "urlweb", urlWeb, 3, 400);
            XMLUtils.endElement(contentHandler, "web");
        }
    }
    
    /**
     * Create some LHEO elements for tag &gt;adresse&lt;
     * <br>Contains the following XML tags:
     * <br>[0,1] &lt;ligne&gt;
     * <br>[0,1] &lt;codepostal&gt;
     * <br>[0,1] &lt;ville&gt;
     * <br>[0,3] &lt;departement&gt;
     * <br>[0,1] &lt;code-INSEE-commune&gt;
     * <br>[0,1] &lt;code-INSEE-canton&gt;
     * <br>[0,1] &lt;region&gt;
     * <br>[0,1] &lt;pays&gt;
     * <br>[0,1] &lt;geolocalisation&gt;
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param address the address. Is mandatory.
     * @param zipCode the zip code. Is mandatory.
     * @param town the town. Is mandatory.
     * @param department the department. Can be null.
     * @param codeINSEEtown the code INSEE for the town. Can be null.
     * @param codeINSEEdistrict the code INSEE for the district. Can be null.
     * @param zone the zone. Can be null.
     * @param country the country. Can be null.
     * @param latitude the latitude. Can be null.
     * @param longitude the longitude. Can be null.
     * @throws SAXException if a saxing exception occurred
     */
    public void createAddressLHEOElements(ContentHandler contentHandler, Content content, String address, String zipCode, String town, String department, String codeINSEEtown, String codeINSEEdistrict, String zone, String country, String latitude, String longitude) throws SAXException
    {
        List<String> lines = StringUtils.isNotBlank(address) ? Splitter.fixedLength(50).splitToList(address) : new ArrayList<>();
        createAddressLHEOElements(contentHandler, content, lines, zipCode, town, department, codeINSEEtown, codeINSEEdistrict, zone, country, latitude, longitude);
    }
    
    /**
     * Create some LHEO elements for tag &gt;adresse&lt;
     * <br>Contains the following XML tags:
     * <br>[0,1] &lt;ligne&gt;
     * <br>[0,1] &lt;codepostal&gt;
     * <br>[0,1] &lt;ville&gt;
     * <br>[0,3] &lt;departement&gt;
     * <br>[0,1] &lt;code-INSEE-commune&gt;
     * <br>[0,1] &lt;code-INSEE-canton&gt;
     * <br>[0,1] &lt;region&gt;
     * <br>[0,1] &lt;pays&gt;
     * <br>[0,1] &lt;geolocalisation&gt;
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param lines the address lines. Is mandatory.
     * @param zipCode the zip code. Is mandatory.
     * @param town the town. Is mandatory.
     * @param department the department. Can be null.
     * @param codeINSEEtown the code INSEE for the town. Can be null.
     * @param codeINSEEdistrict the code INSEE for the district. Can be null.
     * @param zone the zone. Can be null.
     * @param country the country. Can be null.
     * @param latitude the latitude. Can be null.
     * @param longitude the longitude. Can be null.
     * @throws SAXException if a saxing exception occurred
     */
    public void createAddressLHEOElements(ContentHandler contentHandler, Content content, List<String> lines, String zipCode, String town, String department, String codeINSEEtown, String codeINSEEdistrict, String zone, String country, String latitude, String longitude) throws SAXException
    {
        // <ligne>
        int size = lines.size();
        if (size > 4)
        {
            size = 4;
            getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value " + StringUtils.join(lines, "") + " can be set in 4 lines. It will be truncated.");
        }
        
        if (size != 0)
        {
            for (int i = 0; i < size; i++)
            {
                createLHEOElement(contentHandler, content, "ligne", lines.get(i), 1, 50);
            }
        }
        else
        {
            getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The tag adresse must contain at least one line");
        }
        
        // <codepostal>
        createMandatoryLHEOElement(contentHandler, content, "codepostal", zipCode, 5, 5);
        
        // <ville>
        createMandatoryLHEOElement(contentHandler, content, "ville", town, 1, 50);
        
        // <departement>
        createLHEOElement(contentHandler, content, "departement", department, 2, 3);
        
        // <code-INSEE-commune>
        createLHEOElement(contentHandler, content, "code-INSEE-commune", codeINSEEtown, 5, 5);
        
        // <code-INSEE-canton>
        createLHEOElement(contentHandler, content, "code-INSEE-canton", codeINSEEdistrict, 4, 5);
        
        // <region>
        createLHEOElement(contentHandler, content, "region", zone, 2, 2);
        
        // <pays>
        createLHEOElement(contentHandler, content, "pays", country, 2, 2);
        
        // <geolocalisation>
        if (StringUtils.isNotBlank(latitude) && StringUtils.isNotBlank(longitude))
        {
            XMLUtils.startElement(contentHandler, "geolocalisation");
            
            // <latitude>
            createLHEOElement(contentHandler, content, "latitude", latitude, 0, 30);
            
            // <longitude>
            createLHEOElement(contentHandler, content, "longitude", longitude, 0, 30);
            
            XMLUtils.endElement(contentHandler, "geolocalisation");
        }
    }
    
    /**
     * Create LHEO elements for tag &gt;periode&lt;
     * <br>Contains the following XML tags:
     * <br>[1,1] &lt;debut&gt;
     * <br>[1,1] &lt;fin&gt;
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param startDate the start date. Can be null. If null a warning will be logged.
     * @param endDate the end date. Can be null. If null a warning will be logged.
     * @throws SAXException if a saxing exception occurred
     */
    public void createPeriodLHEOElments(ContentHandler contentHandler, Content content, LocalDate startDate, LocalDate endDate) throws SAXException
    {
        if (startDate != null && endDate != null && startDate.isAfter(endDate))
        {
            getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] <periode> : start date should not be greater than end date");
        }
        
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        
        createMandatoryLHEOElement(contentHandler, content, "debut", startDate != null ? formatter.format(startDate) : null, 8, 8);
        
        createMandatoryLHEOElement(contentHandler, content, "fin", endDate != null ? formatter.format(endDate) : null, 8 , 8);
    }
    
    /**
     * Create the mandatory LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param value the value to sax
     * @throws SAXException if a saxing error occurred
     */
    public void createMandatoryLHEOElement(ContentHandler contentHandler, Content content, String elementName, String value) throws SAXException
    {
        createMandatoryLHEOElement(contentHandler, content, elementName, new AttributesImpl(),  value, 0, 0);
    }
    
    /**
     * Create the mandatory LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param value the value to sax
     * @param minLength the min length of the value (0 if there is no min length). If the value length is lower than minLength, a warning will be logged
     * @param maxLength the max length of the value (0 if there is no max length). If the value length is greater than maxLength, a warning will be logged and the value will be troncated
     * @throws SAXException if a saxing error occurred
     */
    public void createMandatoryLHEOElement(ContentHandler contentHandler, Content content, String elementName, String value, int minLength, int maxLength) throws SAXException
    {
        createMandatoryLHEOElement(contentHandler, content, elementName, new AttributesImpl(), value, minLength, maxLength);
    }
    
    /**
     * Create the mandatory LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param attrs the attributes
     * @param value the value to sax
     * @throws SAXException if a saxing error occurred
     */
    public void createMandatoryLHEOElement(ContentHandler contentHandler, Content content, String elementName, AttributesImpl attrs, String value) throws SAXException
    {
        if (StringUtils.isBlank(value))
        {
            getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value for tag " + elementName + " is mandatory");
        }
        else
        {
            createLHEOElement(contentHandler, content, elementName, value, 0, 0);
        }
    }
    
    /**
     * Create the mandatory LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param attrs the attributes
     * @param value the value to sax
     * @param minLength the min length of the value (0 if there is no min length). If the value length is lower than minLength, a warning will be logged
     * @param maxLength the max length of the value (0 if there is no max length). If the value length is greater than maxLength, a warning will be logged and the value will be troncated
     * @throws SAXException if a saxing error occurred
     */
    public void createMandatoryLHEOElement(ContentHandler contentHandler, Content content, String elementName, AttributesImpl attrs, String value, int minLength, int maxLength) throws SAXException
    {
        if (StringUtils.isBlank(value))
        {
            getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value for tag " + elementName + " is mandatory");
        }
        else
        {
            createLHEOElement(contentHandler, content, elementName, value, minLength, maxLength);
        }
    }
    
    /**
     * Create the LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param value the value to sax
     * @throws SAXException if a saxing error occurred
     */
    public void createLHEOElement(ContentHandler contentHandler, Content content, String elementName, String value) throws SAXException
    {
        createLHEOElement(contentHandler, content, elementName, new AttributesImpl(), value, 0, 0);
    }
    
    /**
     * Create the LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param value the value to sax
     * @param minLength the min length of the value (0 if there is no min length). If the value length is lower than minLength, a warning will be logged
     * @param maxLength the max length of the value (0 if there is no max length). If the value length is greater than maxLength, a warning will be logged and the value will be troncated
     * @throws SAXException if a saxing error occurred
     */
    public void createLHEOElement(ContentHandler contentHandler, Content content, String elementName, String value, int minLength, int maxLength) throws SAXException
    {
        createLHEOElement(contentHandler, content, elementName, new AttributesImpl(), value, minLength, maxLength);
    }
    
    /**
     * Create the LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param attrs the attributes
     * @param value the value to sax
     * @throws SAXException if a saxing error occurred
     */
    public void createLHEOElement(ContentHandler contentHandler, Content content, String elementName, AttributesImpl attrs, String value) throws SAXException
    {
        if (StringUtils.isNotBlank(value))
        {
            XMLUtils.createElement(contentHandler, elementName, attrs, value);
        }
    }
    
    /**
     * Create the LHEO element
     * @param contentHandler the content handler
     * @param content the saxed content
     * @param elementName the element name
     * @param attrs the attributes
     * @param value the value to sax
     * @param minLength the min length of the value (0 if there is no min length). If the value length is lower than minLength, a warning will be logged
     * @param maxLength the max length of the value (0 if there is no max length). If the value length is greater than maxLength, a warning will be logged and the value will be troncated
     * @throws SAXException if a saxing error occurred
     */
    public void createLHEOElement(ContentHandler contentHandler, Content content, String elementName, AttributesImpl attrs, String value, int minLength, int maxLength) throws SAXException
    {
        if (StringUtils.isNotBlank(value))
        {
            String computedValue = value;
            int valueLength = computedValue.length();
            if (minLength != 0 && maxLength != 0 && minLength == maxLength)
            {
                if (valueLength < minLength)
                {
                    getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value for element " + elementName + " is supposed to contains exactly " + minLength + " characters.");
                }
                else
                {
                    getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value for element " + elementName + " is supposed to contains exactly " + minLength + " characters. It will be troncated.");
                    computedValue = StringUtils.substring(value, 0, maxLength);
                }
            }
            else
            {
                if (maxLength != 0 && valueLength > maxLength)
                {
                    getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value for element " + elementName + " is not supposed to contains more than " + maxLength + " characters. It will be troncated.");
                    computedValue = StringUtils.substring(value, 0, maxLength);
                }
                
                if (minLength != 0 && valueLength < minLength)
                {
                    getLogger().warn("[" + content.getTitle() + " (" + content.getId() + ")] The value for element " + elementName + " is not supposed to contains less than " + minLength + " characters.");
                }
            }
            
            XMLUtils.createElement(contentHandler, elementName, attrs, computedValue);
        }
    }
}
