/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.xslt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractWrappingAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.repository.AmetysObject;

/**
 * DOM layer on structure of an ODF content.
 * @param <T> The type of wrapped object
 */
public abstract class AbstractODFElement<T extends ProgramItem> extends AbstractWrappingAmetysElement<T>
{
    /** The children depth */
    protected int _depth;
    /** The ODF helper */
    protected ODFHelper _odfHelper;

    /**
     * Constructor.
     * @param odfContent the underlying.
     * @param depth Depth to SAX.
     * @param parent Parent of the element
     * @param odfHelper ODFHelper to resolve children easily
     */
    public AbstractODFElement(T odfContent, int depth, AbstractODFElement<?> parent, ODFHelper odfHelper)
    {
        super(odfContent, parent);
        _depth = depth;
        _odfHelper = odfHelper;
    }

    @Override
    public boolean hasChildNodes()
    {
        if (_depth == 0)
        {
            // Stop recursion
            return false;
        }
        
        return !_odfHelper.getChildProgramItems(_object).isEmpty();
    }
    
    @Override
    public Node getFirstChild()
    {
        List<ProgramItem> children = _odfHelper.getChildProgramItems(_object);
        if (!children.isEmpty())
        {
            return _convertAmetysObjectToElement(children.get(0), this);
        }
        
        return null;
    }
    
    @Override
    public Node getNextSibling()
    {
        if (_parent == null)
        {
            return null;
        }
        
        AmetysObject nextSibling = null;
        
        AbstractODFElement<?> parentElement = (AbstractODFElement<?>) _parent;
        Object parent = parentElement.getWrappedObject();
        if (parent instanceof ProgramPart)
        {
            List<? extends AmetysObject> children = _getProgramPartChildren((ProgramPart) parent);
            nextSibling = _getNextSibling(children);
        }
        else if (parent instanceof Course)
        {
            List<CourseList> children = ((Course) parent).getCourseLists();
            nextSibling = _getNextSibling(children);
        }
        else
        {
            throw new DOMException(DOMException.NOT_SUPPORTED_ERR, "getNextSibling");
        }
        
        if (nextSibling != null)
        {
            return _convertAmetysObjectToElement(nextSibling, parentElement);
        }
        
        return null;
    }
    
    /**
     * Retrieves the children of the given program part
     * @param programPart the program part
     * @return the children of the program part
     */
    protected List<? extends AmetysObject> _getProgramPartChildren(ProgramPart programPart)
    {
        if (programPart instanceof TraversableProgramPart)
        {
            return ((TraversableProgramPart) programPart).getProgramPartChildren();
        }
        else if (programPart instanceof CourseList)
        {
            return ((CourseList) programPart).getCourses();
        }
        else
        {
            // Should never happen, a program part can only be traversable or a course list
            return List.of();
        }
    }
    
    /**
     * The element immediately following this one
     * @param elements the elements that are at the same level as the current one
     * @return The element immediately following this one
     */
    protected AmetysObject _getNextSibling(List<? extends AmetysObject> elements)
    {
        AmetysObject nextSibling = null;
        boolean isNext = false;
        int count = 0;
        
        while (nextSibling == null && count < elements.size())
        {
            if (isNext)
            {
                nextSibling = elements.get(count);
            }
            else if (_object.getId().equals(elements.get(count).getId()))
            {
                isNext = true;
            }
            count++;
        }
        
        return nextSibling;
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> result = new HashMap<>();
        result.put("id", new AmetysAttribute("id", "id", null, _object.getId(), this));
        
        if (StringUtils.isNotEmpty(_object.getCode()))
        {
            result.put("code", new AmetysAttribute("code", "code", null, _object.getCode(), this));
        }
        if (StringUtils.isNotEmpty(_object.getCatalog()))
        {
            result.put("catalog", new AmetysAttribute("catalog", "catalog", null, _object.getCatalog(), this));
        }
        return result;
    }
    
    private Node _convertAmetysObjectToElement(AmetysObject ao, AbstractODFElement<?> parentElement)
    {
        if (ao instanceof SubProgram subProgram)
        {
            return new SubProgramElement(subProgram, _depth, parentElement, _odfHelper);
        }
        
        if (ao instanceof Container container)
        {
            return new ContainerElement(container, _depth, parentElement, _odfHelper);
        }
        
        if (ao instanceof CourseList courseList)
        {
            return new CourseListElement(courseList, _depth, parentElement, _odfHelper);
        }
        
        if (ao instanceof Course course)
        {
            return new CourseElement(course, _depth, (CourseListElement) parentElement, _odfHelper);
        }
        
        return null;
    }
}
