/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.util.Map;

import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Copy updater to update the course holder on a {@link CoursePart}.
 */
public class CoursePartHolderUpdater extends AbstractLogEnabled implements CopyCatalogUpdater
{
    @Override
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        for (Content copiedContent : copiedContents.values())
        {
            if (copiedContent instanceof CoursePart copiedCoursePart)
            {
                Course courseHolder = copiedCoursePart.getCourseHolder();
                if (courseHolder == null)
                {
                    getLogger().warn("The course part '{}' of the catalog '{}' haven't any course holder.", copiedCoursePart.getTitle(), copiedCoursePart.getCatalog());
                }
                else
                {
                    String courseHolderId = courseHolder.getId();
                    Content copiedCourseHolder = copiedContents.entrySet()
                                                               .stream()
                                                               .filter(entry -> courseHolderId.equals(entry.getKey().getId()))
                                                               .findFirst()
                                                               .map(Map.Entry::getValue)
                                                               .orElse(null);
                    if (copiedCourseHolder == null)
                    {
                        getLogger().warn("The course part '{}' of the catalog '{}' can't get the corresponding course holder with the identifier '{}'.", copiedCoursePart.getTitle(), copiedCoursePart.getCatalog(), courseHolder.getId());
                    }
                    else
                    {
                        copiedCoursePart.setValue(CoursePart.COURSE_HOLDER, copiedCourseHolder);
                        copiedCoursePart.saveChanges();
                    }
                }
            }
        }
    }
}
