/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.observation.OdfObservationConstants;
import org.ametys.odf.translation.TranslationHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

import com.opensymphony.workflow.WorkflowException;

/**
 * DAO for manipulating ODF programs.
 *
 */
public class ProgramDAO implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = ProgramDAO.class.getName();
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** ODF helper */
    protected ODFHelper _odfHelper;
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    /** The program translation updater extension point */
    protected ProgramTranslationUpdaterExtensionPoint _programTranslationUpdaterExtensionPoint;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _programTranslationUpdaterExtensionPoint = (ProgramTranslationUpdaterExtensionPoint) manager.lookup(ProgramTranslationUpdaterExtensionPoint.ROLE);
    }
    
    /**
     * Translates a given program in a language
     * @param contentId The program id
     * @param language The language in which to translate
     * @param fullCopy True if full copy
     * @return A Map with id of translated program or an error message
     * @throws Exception if an error occurs
     */
    @Callable (rights = "ODF_Rights_Program_Translate", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, String> translateProgram (String contentId, String language, boolean fullCopy) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        Program program = _resolver.resolveById(contentId);
        
        if (program.getLanguage().equals(language))
        {
            result.put("error", "same-language");
            return result;
        }
        
        Map<Content, Content> translatedContents = new HashMap<>();
        String translatedProgramId = null;
        
        // Get existing translations
        Map<String, String> translations = TranslationHelper.getTranslations(program);
        
        if (!translations.containsKey(language))
        {
            translatedProgramId = copyProgram(program, language, fullCopy, translatedContents);
        }
        else
        {
            translatedProgramId = translations.get(language);
            // Check if content exists
            try
            {
                _resolver.resolveById(translatedProgramId);
                result.put("error", "already-exists");
            }
            catch (UnknownAmetysObjectException e) 
            {
                translatedProgramId = copyProgram(program, language, fullCopy, translatedContents);
            }
        }
        
        linkTranslations(translatedContents);
        
        // Add the workflow step, send notifications and extract outgoing references
        for (Content newContent : translatedContents.values())
        {
            if (newContent instanceof WorkflowAwareContent newWorkflowAwareContent)
            {
                _contentWorkflowHelper.doAction(newWorkflowAwareContent, getCopyActionId());
            }
        }
        
        result.put("translated-program-id", translatedProgramId);
        
        return result;
    }

    /**
     * Copy the given {@link Program} for translation
     * @param program The program to copy
     * @param language The language in which to translate
     * @param fullCopy Set to <code>true</code> to copy the sub-structure
     * @param copiedContents the initial contents with their copied content
     * @return The created content identifier
     * @throws WorkflowException If an error occurred
     */
    protected String copyProgram(Program program, String language, boolean fullCopy, Map<Content, Content> copiedContents) throws WorkflowException
    {
        Program copiedProgram = _odfHelper.copyProgramItem(program, null, language, null, fullCopy, copiedContents);
        
        Set<String> copyUpdaters = _programTranslationUpdaterExtensionPoint.getExtensionsIds();
        for (String updaterId : copyUpdaters)
        {
            // Call updaters after copy of program
            ProgramTranslationUpdater updater = _programTranslationUpdaterExtensionPoint.getExtension(updaterId);
            updater.updateContents(null, null, copiedContents, null);
        }
        
        return copiedProgram.getId();
    }
    
    /**
     * Store links to the other translations for all copied contents
     * @param translatedContents The translated contents
     */
    protected void linkTranslations(Map<Content, Content> translatedContents)
    {
        for (Entry<Content, Content> entry : translatedContents.entrySet())
        {
            ModifiableContent originalContent = (ModifiableContent) entry.getKey();
            ModifiableContent translatedContent = (ModifiableContent) entry.getValue();
            
            linkTranslations(originalContent, translatedContent);
            
            originalContent.saveChanges();
            translatedContent.saveChanges();
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_CONTENT, originalContent);
            
            _observationManager.notify(new Event(OdfObservationConstants.ODF_CONTENT_TRANSLATED, _getCurrentUser(), eventParams));
        }
    }
    
    /**
     * Store links to the other translations in all the translated objects.
     * @param originalContent The original content
     * @param translatedContent The translated content
     */
    protected void linkTranslations(ModifiableContent originalContent, ModifiableContent translatedContent)
    {
        Map<String, String> translations = TranslationHelper.getTranslations(originalContent);
        
        // Add the original and translated references.
        translations.put(originalContent.getLanguage(), originalContent.getId());
        translations.put(translatedContent.getLanguage(), translatedContent.getId());
        
        for (String contentId : translations.values())
        {
            ModifiableContent content = _resolver.resolveById(contentId);
            
            Map<String, String> otherTranslations = new HashMap<>(translations);
            otherTranslations.remove(content.getLanguage());
            
            TranslationHelper.setTranslations(content, otherTranslations);
        }
    }
    
    /**
     * Provides the current user.
     * @return the identity which cannot be <code>null</code>.
     */
    protected UserIdentity _getCurrentUser()
    {
        return _currentUserProvider.getUser();
    }
    
    /**
     * Get the workflow action id for copy.
     * @return The workflow action id
     */
    protected int getCopyActionId()
    {
        return 210;
    }
}
