/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobExecutionContext;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * Scheduler to purge orphan course parts.
 */
public class CoursePartPurgeSchedulable extends AbstractStaticSchedulable
{
    private AmetysObjectResolver _resolver;
    private ContentDAO _contentDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        // Getting all orphan course parts (linked to any course)
        String xPathQuery = QueryHelper.getXPathQuery(null, CoursePartFactory.COURSE_PART_NODETYPE, null);
        // Search course parts only
        List<String> contentIdsToDelete = _resolver.<CoursePart>query(xPathQuery)
            .stream()
            // Verify if the content has no parent courses
            .filter(coursePart -> coursePart.getCourses().isEmpty())
            // Twice...
            .filter(coursePart -> {
                String query = QueryHelper.getXPathQuery(null, CourseFactory.COURSE_NODETYPE, new StringExpression(Course.CHILD_COURSE_PARTS, Operator.EQ, coursePart.getId()));
                return !_resolver.query(query).iterator().hasNext();
            })
            // Log
            .peek(coursePart -> getLogger().info("The purge will delete the content {}.", coursePart.toString()))
            // Get the identifier
            .map(Content::getId)
            // Collect
            .collect(Collectors.toList());
        
        // Remove the course parts
        _contentDAO.deleteContents(contentIdsToDelete, true);
    }
}
