/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.RichTextHelper;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.View;

/**
 * Generator for orientation paths in program.
 */
public class ProgramOrientationPathsGenerator extends ServiceableGenerator
{
    // FIXME Accept all view names starting with "main" to be able to make the ODF and ODF orientation skins coexist (whereas they use different main views)
    private static final Pattern __ALLOWED_VIEW_NAMES = Pattern.compile("main[a-z\\-]*");
    // Max number of characters for description per year of duration
    private static int _MAX_DESC_LENGTH_PER_YEAR = 160;

    private AmetysObjectResolver _resolver;
    private ContentTypesHelper _cTypesHelper;
    private ODFHelper _odfHelper;
    private RichTextHelper _richTextHelper;
    
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _cTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _richTextHelper = (RichTextHelper) smanager.lookup(RichTextHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Content content = (Content) request.getAttribute(Content.class.getName());
        
        if (content == null)
        {
            String contentId = request.getParameter("contentId");
            if (StringUtils.isBlank(contentId))
            {
                throw new IllegalArgumentException("Content is missing in request attribute or parameters");
            }
            
            content = _resolver.resolveById(contentId);
        }

        if (!(content instanceof AbstractProgram abstractProgram))
        {
            throw new IllegalArgumentException("Cannot get orientation paths of a non abstract program '" + content.getId() + "'");
        }
        
        String viewName = parameters.getParameter("viewName", StringUtils.EMPTY);
        String fallbackViewName = parameters.getParameter("fallbackViewName", StringUtils.EMPTY);
        
        View view = _cTypesHelper.getViewWithFallback(viewName, fallbackViewName, content);
        
        // only allow view starting with "main" to optimize performances when structure is not needed
        if (view == null || !__ALLOWED_VIEW_NAMES.matcher(view.getName()).matches())
        {
            contentHandler.startDocument();
            XMLUtils.createElement(contentHandler, "orientation-paths");
            contentHandler.endDocument();
            return;
        }
        
        // Extract all years first to determine min and max levels
        List<YearContainer> allYears = _extractAllYears(abstractProgram);
        
        // Analyze the program structure to extract orientation paths
        List<OrientationPath> orientationPaths = _computePaths(allYears);
        
        contentHandler.startDocument();
        
        int minYear = orientationPaths.stream()
                                      .mapToInt(OrientationPath::minYear)
                                      .min()
                                      .orElse(-1);

        int maxYear = orientationPaths.stream()
                                      .mapToInt(OrientationPath::maxYear)
                                      .max()
                                      .orElse(-1);

        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("minYear", String.valueOf(minYear));
        attr.addCDATAAttribute("maxYear", String.valueOf(maxYear));

        XMLUtils.startElement(contentHandler, "orientation-paths", attr);

        // Generate SAX events for each orientation path
        for (OrientationPath path : orientationPaths)
        {
            _saxOrientationPath(path);
        }
        
        XMLUtils.endElement(contentHandler, "orientation-paths");
        
        contentHandler.endDocument();
    }
    
    private List<YearContainer> _extractAllYears(AbstractProgram abstractProgram)
    {
        List<YearContainer> result = new ArrayList<>();
        
        Set<Container> years = _odfHelper.getYears(abstractProgram);
        Set<String> yearIds = years.stream().map(Container::getId).collect(Collectors.toSet());

        for (Container year : years)
        {
            // Extract year level from period (annee1 -> 1, annee2 -> 2, etc.)
            int yearLevel = -1;
            
            try
            {
                String period = year.getValue("period/code");
                if (StringUtils.isNotBlank(period) && period.startsWith("annee") && period.length() > 5)
                {
                    // Extract number from "anneeX" pattern
                    String numberStr = period.substring(5); // Skip "annee"
                    yearLevel = Integer.parseInt(numberStr);
                }
            }
            catch (Exception e)
            {
                getLogger().warn("Failed to extract year level from container " + year.getId(), e);
            }

            if (yearLevel > 0)
            {
                // Filter previousYears and nextYears to only include those present in the current program part
                
                ContentValue[] previousYears = year.getValue("previousYears");
                List<ContentValue> currentPreviousYears = previousYears != null ? Arrays.stream(previousYears).filter(y -> yearIds.contains(y.getContentId())).toList() : List.of();

                ContentValue[] nextYears = year.getValue("nextYears");
                List<ContentValue> currentNextYears = nextYears != null ? Arrays.stream(nextYears).filter(y -> yearIds.contains(y.getContentId())).toList() : List.of();

                result.add(new YearContainer(year, yearLevel, (TraversableProgramPart) _odfHelper.getParentProgramItem(year, abstractProgram), currentPreviousYears, currentNextYears));
            }
        }
        
        return result;
    }
    
    private List<OrientationPath> _computePaths(List<YearContainer> allYears)
    {
        // Build a map for quick lookup by container ID
        Map<String, YearContainer> yearsMap = allYears.stream()
                                                      .collect(Collectors.toMap(y -> y.container().getId(), y -> y));
        
        // Find starting years: those without previousYears and with a valid period
        List<YearContainer> startingYears = allYears.stream()
                                                    .filter(y -> y.previousYears().isEmpty() && y.yearLevel() > 0)
                                                    .toList();
        
        // Build paths starting from each starting year
        List<OrientationPath> paths = new ArrayList<>();
        for (YearContainer startYear : startingYears)
        {
            OrientationPath path = _buildPath(startYear, yearsMap);
            
            if (path != null)
            {
                paths.add(path);
            }
        }
        
        return paths;
    }
    
    private OrientationPath _buildPath(YearContainer fromYear, Map<String, YearContainer> yearsMap)
    {
        List<PathPart> parts = _getParts(fromYear, yearsMap, new HashSet<>());
        
        if (parts == null || parts.isEmpty())
        {
            return null;
        }
        
        int minYear = parts.stream().mapToInt(PathPart::minYear).min().orElse(-1);
        int maxYear = parts.stream().mapToInt(PathPart::maxYear).max().orElse(-1);

        return new OrientationPath(minYear, maxYear, parts);
    }
        
    private List<PathPart> _getParts(YearContainer fromYear, Map<String, YearContainer> yearsMap, Set<String> visitedYears)
    {        
        String currentYearId = fromYear.container().getId();
        
        // Check for cycles
        if (visitedYears.contains(currentYearId))
        {
            getLogger().warn("Cycle detected at year " + currentYearId + ". Stopping path construction.");
            return null;
        }
        
        visitedYears.add(currentYearId);
        
        // Group consecutive years under the same parent using nextYears
        List<YearContainer> groupedYears = new ArrayList<>();
        groupedYears.add(fromYear);

        YearContainer lastYear = fromYear;

        // Follow nextYears chain as long as they stay in the same parent
        boolean keepGrouping = true;
        while (keepGrouping)
        {
            YearContainer currentLastYear = lastYear;
            
            // Check if there's exactly one next year and in the same parent
            List<ContentValue> nextYears = currentLastYear.nextYears();
            if (nextYears.size() == 1)
            {
                YearContainer nextYear = yearsMap.get(nextYears.get(0).getContentId());
                if (nextYear != null && nextYear.parent().getId().equals(currentLastYear.parent().getId()))
                {
                    groupedYears.add(nextYear);
                    visitedYears.add(nextYear.container().getId());
                    lastYear = nextYear;
                    continue;
                }
            }
            
            // No single next year in same parent - stop grouping
            keepGrouping = false;
        }
        
        // Check if the grouped years represent ALL years in the parent
        TraversableProgramPart parent = fromYear.parent();
        boolean groupContainsAllYearsInParent = groupedYears.size() == parent.getProgramPartChildren().size();
        
        List<PathPart> parts = new ArrayList<>();

        if (groupContainsAllYearsInParent)
        {
            // Create a PathPart for the grouped years
            parts.add(_createGroupedPathPart(groupedYears, parent));
        }
        else
        {
            for (YearContainer year : groupedYears)
            {
                // Create a PathPart for each year individually
                parts.add(_createSingleYearPathPart(year));
            }
        }
        
        // Get all next years from the last year in the group
        List<String> allNextYearIds = lastYear.nextYears().stream()
                                              .map(cv -> cv.getContentId())
                                              .toList();
        
        // Continue building path for each next year (handles bifurcations)
        for (String nextYearId : allNextYearIds)
        {
            YearContainer nextYear = yearsMap.get(nextYearId);
            if (nextYear != null && !visitedYears.contains(nextYearId))
            {
                parts.addAll(_getParts(nextYear, yearsMap, visitedYears));
            }
        }
        
        return parts;
    }
    
    private PathPart _createSingleYearPathPart(YearContainer year)
    {
        String id = "part-" + year.container().getId();
        int yearLevel = year.yearLevel();
        RichText desc = year.container().getDescription();
        String description = desc != null ? _richTextHelper.richTextToString(desc, _MAX_DESC_LENGTH_PER_YEAR) : "";
        
        List<String> nextParts = year.nextYears().stream().map(y -> "part-" + y.getContentId()).toList();

        return new PathPart(id, year.parent().getId(), _getParentType(year.parent()), year.container().getTitle(), description, yearLevel, yearLevel, List.of(year.container()), nextParts, false);
    }
    
    private PathPart _createGroupedPathPart(List<YearContainer> years, ProgramPart parent)
    {
        YearContainer firstYear = years.get(0);
        YearContainer lastYear = years.get(years.size() - 1);
        String id = "part-" + firstYear.container().getId();
        int minYear = firstYear.yearLevel();
        int maxYear = lastYear.yearLevel();
        
        RichText desc;
        if (parent instanceof AbstractProgram abstractProgram)
        {
            desc = abstractProgram.getPresentation();
        }
        else
        {
            desc = ((Container) parent).getDescription();
        }

        // Adapt description length based on duration
        String description = desc != null ? _richTextHelper.richTextToString(desc, (maxYear - minYear + 1) * _MAX_DESC_LENGTH_PER_YEAR) : "";
        
        List<Container> containers = years.stream().map(y -> y.container()).toList();
        List<String> nextParts = lastYear.nextYears().stream().map(y -> "part-" + y.getContentId()).toList();

        return new PathPart(id, parent.getId(), _getParentType(parent), ((Content) parent).getTitle(), description, minYear, maxYear, containers, nextParts, true);
    }
    
    private String _getParentType(ProgramPart parent)
    {
        if (parent instanceof SubProgram)
        {
            return "subProgram";
        }
        else if (parent instanceof Program)
        {
            return "program";
        }
        else
        {
            return "container";
        }
    }
    
    private void _saxOrientationPath(OrientationPath path) throws SAXException
    {
        AttributesImpl pathAttrs = new AttributesImpl();
        pathAttrs.addCDATAAttribute("minYear", String.valueOf(path.minYear()));
        pathAttrs.addCDATAAttribute("maxYear", String.valueOf(path.maxYear()));
        pathAttrs.addCDATAAttribute("duration", String.valueOf(path.maxYear() - path.minYear() + 1));
        XMLUtils.startElement(contentHandler, "path", pathAttrs);
        
        for (PathPart part : path.parts())
        {
            _saxPathPart(part);
        }
        
        XMLUtils.endElement(contentHandler, "path");
    }
    
    private void _saxPathPart(PathPart part) throws SAXException
    {
        AttributesImpl partAttrs = new AttributesImpl();
        
        partAttrs.addCDATAAttribute("id", part.id());
        partAttrs.addCDATAAttribute("parentId", part.parentId());
        partAttrs.addCDATAAttribute("parentType", part.parentType());
        partAttrs.addCDATAAttribute("minYear", String.valueOf(part.minYear()));
        partAttrs.addCDATAAttribute("maxYear", String.valueOf(part.maxYear()));
        partAttrs.addCDATAAttribute("duration", String.valueOf(part.maxYear() - part.minYear() + 1));
        partAttrs.addCDATAAttribute("isGrouped", String.valueOf(part.isGrouped()));
        
        XMLUtils.startElement(contentHandler, "part", partAttrs);
        
        XMLUtils.createElement(contentHandler, "title", Objects.toString(part.title(), ""));
        XMLUtils.createElement(contentHandler, "description", Objects.toString(part.description(), ""));
        
        // Generate container references
        XMLUtils.startElement(contentHandler, "containers");
        
        for (Container container : part.containers())
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addCDATAAttribute("id", container.getId());
            XMLUtils.startElement(contentHandler, "container", attr);

            ContentValue[] pathways = container.getValue("pathways", false, new ContentValue[0]);
            for (ContentValue pathway : pathways)
            {
                AttributesImpl pathwayAttr = new AttributesImpl();
                pathwayAttr.addCDATAAttribute("id", pathway.getContentId());
                XMLUtils.createElement(contentHandler, "pathway", pathwayAttr);
            }
            
            XMLUtils.endElement(contentHandler, "container");
        }
        
        XMLUtils.endElement(contentHandler, "containers");
        
        // Generate next parts references
        XMLUtils.startElement(contentHandler, "next-parts");
        
        for (String nextId : part.nextPartIds())
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addCDATAAttribute("ref", nextId);
            XMLUtils.createElement(contentHandler, "part", attr);
        }

        XMLUtils.endElement(contentHandler, "next-parts");
        
        XMLUtils.endElement(contentHandler, "part");
    }
    
    private record YearContainer(Container container, int yearLevel, TraversableProgramPart parent, List<ContentValue> previousYears, List<ContentValue> nextYears) { /* empty */ }
    
    /**
     * Represents a part of a path, which can span multiple consecutive years under the same parent.
     * Years are grouped together if they share the same parent.
     * @param id unique identifier for this path part
     * @param parentId ID of the parent (Program or SubProgram) containing these years
     * @param parentType the type of the parent
     * @param title title describing this part of the path
     * @param description description of this part
     * @param minYear minimum year level in this path
     * @param maxYear maximum year level in this path
     * @param containers list of containers that make up this part
     * @param nextPartIds list of possible next path part IDs (for bifurcations)
     * @param isGrouped indicates if this part represents grouped years
     */
    public record PathPart(String id, String parentId, String parentType, String title, String description, int minYear, int maxYear, List<Container> containers, List<String> nextPartIds, boolean isGrouped) { /* empty */ }
    
    /**
     * Represents a complete orientation path in the program.
     * @param minYear minimum year level in this path
     * @param maxYear maximum year level in this path
     * @param parts ordered list of path parts that make up this path
     */
    public record OrientationPath(int minYear, int maxYear, List<PathPart> parts) { /* empty */ }
}
