/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.lheo;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.RichText;
import org.ametys.cms.data.RichTextHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Manager to generate programs to LHEO XML
 */
public class ExportToLHEOManager extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = ExportToLHEOManager.class.getName();
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The rich text helper */
    protected RichTextHelper _richTextHelper;
    
    /** The LHEO utils */
    protected LHEOUtils _lheoUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _richTextHelper = (RichTextHelper) manager.lookup(RichTextHelper.ROLE);
        _lheoUtils = (LHEOUtils) manager.lookup(LHEOUtils.ROLE);
    }
    
    /**
     * Sax the LHEO xml for the list of programs
     * @param contentHandler the content handler
     * @param programs the list of program to sax
     * @throws SAXException if a saxing exception occurred
     */
    public void saxLHEO(ContentHandler contentHandler, List<AbstractProgram> programs) throws SAXException
    {
        saxLHEO(contentHandler, programs, new HashMap<>());
    }
    
    /**
     * Sax the LHEO xml for the list of programs
     * @param contentHandler the content handler
     * @param programs the list of program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    public void saxLHEO(ContentHandler contentHandler, List<AbstractProgram> programs, Map<String, Object> additionalParameters) throws SAXException
    {
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("", "http://www.lheo.org/2.2");
            
        XMLUtils.startElement(contentHandler, "lheo");
        _saxOffers(contentHandler, programs, additionalParameters);
        XMLUtils.endElement(contentHandler, "lheo");
        
        contentHandler.endDocument();
    }

    /**
     * Sax the XML tag &lt;offres&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,N] &lt;formation&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param programs the list of program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxOffers(ContentHandler contentHandler, List<AbstractProgram> programs, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "offres");
        _saxPrograms(contentHandler, programs, additionalParameters);
        _saxOffersExtras(contentHandler, programs, additionalParameters);
        XMLUtils.endElement(contentHandler, "offres");
    }

    /**
     * Sax for each program a XML tag &lt;formation&gt;
     * @param contentHandler the content handler
     * @param programs the list of program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxPrograms(ContentHandler contentHandler, List<AbstractProgram> programs, Map<String, Object> additionalParameters) throws SAXException
    {
        for (AbstractProgram program : programs)
        {
            _saxProgram(contentHandler, program, additionalParameters);
        }
    }
    
    /**
     * Sax a XML tag &lt;formation&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;domaine-formation&gt;
     * <br>[1,1] &lt;intitule-formation&gt;
     * <br>[1,1] &lt;objectif-formation&gt;
     * <br>[1,1] &lt;resultats-attendus&gt;
     * <br>[1,1] &lt;contenu-formation&gt;
     * <br>[1,1] &lt;certifiante&gt;
     * <br>[1,1] &lt;contact-formation&gt;
     * <br>[1,1] &lt;parcours-de-formation&gt;
     * <br>[1,1] &lt;code-niveau-entree&gt;
     * <br>[0,1] &lt;objectif-general-formation&gt;
     * <br>[0,5] &lt;certification&gt;
     * <br>[0,1] &lt;code-niveau-sortie&gt;
     * <br>[0,1] &lt;url-formation&gt;
     * <br>[1,N] &lt;action&gt;
     * <br>[1,1] &lt;organisme-formation-responsable&gt;
     * <br>[0,1] &lt;identifiant-module&gt;
     * <br>[0,1] &lt;positionnement&gt;
     * <br>[0,1] &lt;sous-modules&gt;
     * <br>[0,1] &lt;modules-prerequis&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgram(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "formation", _getProgramAttributes(program, additionalParameters));
        
        // <domaine-formation>
        _saxProgramDomain(contentHandler, program, additionalParameters);
        
        // <intitule-formation>
        _saxProgramTitle(contentHandler, program, additionalParameters);
        
        // <objectif-formation>
        _saxProgramObjectives(contentHandler, program, additionalParameters);
        
        // <resultats-attendus>
        _saxProgramExpectedResults(contentHandler, program, additionalParameters);
        
        // <contenu-formation>
        _saxProgramPresentation(contentHandler, program, additionalParameters);
        
        // <certifiante>
        _saxProgramCertifying(contentHandler, program, additionalParameters);
        
        // <contact-formation>
        _saxProgramContact(contentHandler, program, additionalParameters);
        
        // <parcours-de-formation>
        _saxProgramPath(contentHandler, program, additionalParameters);
        
        // <code-niveau-entree>
        _saxProgramEducationEntryLevel(contentHandler, program, additionalParameters);
        
        // <certification>
        _saxProgramCertification(contentHandler, program, additionalParameters);

        // <action>
        _saxProgramAction(contentHandler, program, additionalParameters);
        
        // <organisme-formation-responsable>
        _saxProgramResponsibleOrgUnit(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxProgramsExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "formation");
    }

    /**
     * Get attribute for XML tag &lt;formation&gt;
     * @param program the program
     * @param additionalParameters the additional parameters
     * @return the attributes for XML tag &lt;formation&gt;
     */
    protected AttributesImpl _getProgramAttributes(AbstractProgram program, Map<String, Object> additionalParameters)
    {
        AttributesImpl attributesImpl = new AttributesImpl();
        attributesImpl.addCDATAAttribute("numero", program.getCode());
        
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        
        ZonedDateTime creationDate = program.getCreationDate();
        LocalDate creationDateToLocalDate = DateUtils.asLocalDate(creationDate.toInstant());
        attributesImpl.addCDATAAttribute("datecrea", formatter.format(creationDateToLocalDate));
        
        ZonedDateTime lastModificationDate = program.getLastModified();
        LocalDate lastModificationDateToLocalDate = DateUtils.asLocalDate(lastModificationDate.toInstant());
        attributesImpl.addCDATAAttribute("datemaj", formatter.format(lastModificationDateToLocalDate));
        
        return attributesImpl;
    }
    
    /**
     * Sax the XML tag &lt;domaine-formation&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,5] &lt;cpde-FORMACODE&gt;
     * <br>[0,3] &lt;code-NSF&gt;
     * <br>[0,5] &lt;code-ROME&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramDomain(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "domaine-formation");
        
        // <code-FORMACODE>
        _saxFORMACODE(contentHandler, program, additionalParameters);
        
        // <code-NSF>
        _saxNSF(contentHandler, program, additionalParameters);
        
        // <code-ROME>
        _saxROME(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxProgramDomainExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "domaine-formation");
    }

    /**
     * Sax the XML tag &lt;code-FORMACODE&gt;
     * <br>The code contains exactly 5 characters
     * <br>The tag must contains the attribute "ref"
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxFORMACODE(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("ref", "V12"); // The value is fixed for the default implementation
        String[] formacodes = program.getFORMACODE();
        if (formacodes.length > 5)
        {
            getLogger().warn("[" + program.getTitle() + " (" + program.getId() + ")] The LHEO format can have only 5 FORMACODE");
        }
        
        for (String code : formacodes)
        {
            if (StringUtils.isNotBlank(code))
            {
                _lheoUtils.createLHEOElement(contentHandler, program, "code-FORMACODE", attrs, code, 5, 5);
            }
        }
    }

    /**
     * Sax the XML tag &lt;code-NSF&gt;
     * <br>The code contains exactly 3 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxNSF(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String nsfCodeId = program.getNSFCode();
        if (StringUtils.isNotBlank(nsfCodeId))
        {
            Content nsfCodeContent = _resolver.resolveById(nsfCodeId);
            String code = nsfCodeContent.getValue("code");
            if (StringUtils.isNotBlank(code))
            {
                _lheoUtils.createLHEOElement(contentHandler, program, "code-NSF", code, 3, 3);
            }
        }
    }

    /**
     * Sax the XML tag &lt;code-ROME&gt;
     * <br>The code contains exactly 5 characters
     * <br>The tag can have the attribute "ref". If not, the default value is "V3"
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxROME(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String[] romeCodes = program.getRomeCode();
        if (romeCodes.length > 5)
        {
            getLogger().warn("[" + program.getTitle() + " (" + program.getId() + ")] The LHEO format can have only 5 rome code");
        }
        
        for (String romeCodeId : romeCodes)
        {
            Content romeCodeContent = _resolver.resolveById(romeCodeId);
            String code = romeCodeContent.getValue("code");
            if (StringUtils.isNotBlank(code))
            {
                _lheoUtils.createLHEOElement(contentHandler, program, "code-ROME", code, 5, 5);
            }
        }
    }
    
    /**
     * Sax for program domain the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramDomainExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;intitule-formation&gt;
     * <br>The value contains between 1 to 255 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramTitle(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "intitule-formation", program.getTitle(), 1, 255);
    }
    
    /**
     * Sax the XML tag &lt;objectif-formation&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramObjectives(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String richText2String = _richText2String(program.getObjectives());
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "objectif-formation", richText2String, 1, 3000);
    }
    
    /**
     * Sax the XML tag &lt;resultats-attendus&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramExpectedResults(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String richText2String = _richText2String(program.getExpectedResults());
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "resultats-attendus", richText2String, 1, 3000);
    }
    
    /**
     * Sax the XML tag &lt;contenu-formation&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramPresentation(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String richText2String = _richText2String(program.getPresentation());
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "contenu-formation", richText2String, 1, 3000);
    }
    
    /**
     * Sax the XML tag &lt;certifiante&gt;
     * <br>0 for false and 1 for true
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramCertifying(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        boolean certifying = program.isCertifying();
        String certifyingAsString = certifying ? "1" : "0";
        _lheoUtils.createLHEOElement(contentHandler, program, "certifiante", certifyingAsString);
    }
    
    /**
     * Sax the XML tag &lt;contact-formation&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;coordonnees&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramContact(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "contact-formation");
        
        // <coordonnees>
        _saxProgramContactCoordonnees(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxProgramContactExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "contact-formation");
    }

    /**
     * Sax for program contact the XML tag &lt;coordonnees&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * <br>[0,1] &lt;adresse&gt;
     * <br>[0,1] &lt;telfixe&gt;
     * <br>[0,1] &lt;portable&gt;
     * <br>[0,1] &lt;fax&gt;
     * <br>[0,1] &lt;courriel&gt;
     * <br>[0,1] &lt;web&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramContactCoordonnees(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "coordonnees");
        
        Optional<Person> personOpt = _getFirstContact(program);
        if (personOpt.isPresent())
        {
            Person person = personOpt.get();
            _saxPersonCoordinate(contentHandler, person, additionalParameters);
        }
        
        XMLUtils.endElement(contentHandler, "coordonnees");
    }
    
    /**
     * Sax for program contact the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramContactExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;parcours-de-formation&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramPath(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // In default implementation, the path value is always 1 ('En groupe')
        _lheoUtils.createLHEOElement(contentHandler, program, "parcours-de-formation", "1"); 
    }
    
    /**
     * Sax the XML tag &lt;code-niveau-entree&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramEducationEntryLevel(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String[] educationLevelEntries = program.getEducationLevelEntry();
        
        String educationLevelEntryForLHEO = null;
        if (educationLevelEntries.length > 0)
        {
            String educationLevelEntryId = educationLevelEntries[0]; // Take the first
            Content educationLevelEntryContent = _resolver.resolveById(educationLevelEntryId);
            
            String code = educationLevelEntryContent.getValue("code");
            educationLevelEntryForLHEO = _convertEducationEntryLevel2LHEO(code);
        }
        
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "code-niveau-entree", educationLevelEntryForLHEO); 
    }
    
    /**
     * Convert the Ametys education entry level code to LHEO
     * @param code the ametys code
     * @return the LHEO key value
     */
    protected String _convertEducationEntryLevel2LHEO(String code)
    {
        // No code or "Bac + 1" or "Inconnu" or "Inferieur ou égal au baccalauréat"
        if (StringUtils.isBlank(code) || "1".equals(code) || "9".equals(code) || "0".equals(code))
        {
            // "information non communiquée"
            return "0";
        }
        // "Bac + 2"
        else if ("2".equals(code))
        {
            // "niveau III (BTS, DUT)"
            return "6";
        }
        // "Bac + 3"
        else if ("3".equals(code))
        {
            // "niveau II (licence ou maîtrise universitaire)"
            return "7";
        }
        // "Bac + 4" or "Bac + 5" or "Bac + 6" or "Bac + 7" or "Bac + 8"
        else
        {
            // "niveau I (supérieur à la maîtrise)"
            return "8";
        }
    }
    
    /**
     * Sax the XML tag &lt;certification&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,1] &lt;code-RNCP&gt;
     * <br>[0,1] &lt;code-CERTIFINFO&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramCertification(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "certification");
        
        // <code-RNCP>
        _saxProgramRNCPCode(contentHandler, program, additionalParameters);
        
        // <code-CERTIFINFO>
        _saxProgramCERTIFINFOCode(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxCertificationExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "certification");
    }
    
    /**
     * Sax the XML tag &lt;action&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;rythme-formation&gt;
     * <br>[1,10] &lt;code-public-vise&gt;
     * <br>[0,1] &lt;info-public-vise&gt;
     * <br>[1,1] &lt;niveau-entree-obligatoire&gt;
     * <br>[1,1] &lt;modalites-alternance&gt;
     * <br>[1,1] &lt;modalites-enseignement&gt;
     * <br>[1,1] &lt;conditions-specifiques&gt;
     * <br>[1,1] &lt;prise-en-charge-frais-possible&gt;
     * <br>[1,1] &lt;lieu-de-formation&gt;
     * <br>[1,1] &lt;modalites-entrees-sorties&gt;
     * <br>[0,1] &lt;url-action&gt;
     * <br>[1,N] &lt;session&gt;
     * <br>[0,1] &lt;adresse-information&gt;
     * <br>[0,3] &lt;date-information&gt;
     * <br>[0,1] &lt;restauration&gt;
     * <br>[0,1] &lt;hebergement&gt;
     * <br>[0,1] &lt;transport&gt;
     * <br>[0,1] &lt;acces-handicapes&gt;
     * <br>[0,1] &lt;langue-formation&gt;
     * <br>[0,1] &lt;modalites-recrutement&gt;
     * <br>[0,1] &lt;modalites-pedagogiques&gt;
     * <br>[0,5] &lt;code-modalite-pedagogique&gt;
     * <br>[0,1] &lt;frais-restants&gt;
     * <br>[0,1] &lt;code-perimetre-recrutement&gt;
     * <br>[0,1] &lt;infos-perimetre-recrutement&gt;
     * <br>[0,1] &lt;prix-horaire-TTC&gt;
     * <br>[0,1] &lt;prix-total-TTC&gt;
     * <br>[0,1] &lt;duree-indicative&gt;
     * <br>[0,1] &lt;nombre-heures-centre&gt;
     * <br>[0,1] &lt;nombre-heures-entreprise&gt;
     * <br>[0,1] &lt;nombre-heures-total&gt;
     * <br>[0,1] &lt;detail-conditions-prise-en-charge&gt;
     * <br>[0,1] &lt;conventionnement&gt;
     * <br>[0,1] &lt;duree-conventionnee&gt;
     * <br>[0,1] &lt;organisme-formateur&gt;
     * <br>[0,8] &lt;organisme-financeur&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramAction(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "action", _getActionAttributes(program, additionalParameters));
        
        // <rythme-formation>
        _saxProgramTiming(contentHandler, program, additionalParameters);
        
        // <code-public-vise>
        _saxProgramTargetAudience(contentHandler, program, additionalParameters);
        
        // <niveau-entree-obligatoire>
        _saxProgramMandatoryEntryLevel(contentHandler, program, additionalParameters);
        
        // <modalites-alternance>
        _saxProgramApprenticeshipModalities(contentHandler, program, additionalParameters);
        
        // <modalites-enseignement>
        _saxProrgamDistanceLearning(contentHandler, program, additionalParameters);
        
        // <conditions-specifiques>
        _saxProgramNeededPrerequisite(contentHandler, program, additionalParameters);
        
        // <prise-en-charge-frais-possible>
        _saxProgramCostBearing(contentHandler, program, additionalParameters);
        
        // <lieu-de-formation>
        _saxProgramPlaces(contentHandler, program, additionalParameters);
        
        // <modalites-entrees-sorties>
        _saxProgramEntryExitModalities(contentHandler, program, additionalParameters);
        
        // <session>
        _saxProgramSession(contentHandler, program, additionalParameters);
        
        // <langue-formation>
        _saxProgramEducationLanguage(contentHandler, program, additionalParameters);

        // <modalites-recrutement>
        _saxProgramAccessCondition(contentHandler, program, additionalParameters);
        
        // <code-modalite-pedagogique>
        _saxProgramEducationalModalities(contentHandler, program, additionalParameters);
        
        // <organisme-formateur>
        _saxProgramOrgUnitFormer(contentHandler, program, additionalParameters);
        
        // <organisme-financeur>
        _saxProgramOrgUnitFunder(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxActionExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "action");
    }
    
    /**
     * Get attribute for XML tag &lt;action&gt;
     * @param program the program
     * @param additionalParameters the additional parameters
     * @return the attributes for XML tag &lt;action&gt;
     */
    protected AttributesImpl _getActionAttributes(AbstractProgram program, Map<String, Object> additionalParameters)
    {
        AttributesImpl attributesImpl = new AttributesImpl();
        attributesImpl.addCDATAAttribute("numero", program.getCode());
        
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        
        ZonedDateTime creationDate = program.getCreationDate();
        LocalDate creationDateToLocalDate = DateUtils.asLocalDate(creationDate.toInstant());
        attributesImpl.addCDATAAttribute("datecrea", formatter.format(creationDateToLocalDate));
        
        ZonedDateTime lastModificationDate = program.getLastModified();
        LocalDate lastModificationDateToLocalDate = DateUtils.asLocalDate(lastModificationDate.toInstant());
        attributesImpl.addCDATAAttribute("datemaj", formatter.format(lastModificationDateToLocalDate));
        
        return attributesImpl;
    }

    /**
     * Sax the XML tag &lt;rythme-formation&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramTiming(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // In default implementation, the program timing value is always "Temps plein"
        _lheoUtils.createLHEOElement(contentHandler, program, "rythme-formation", "Temps plein");
    }

    /**
     * Sax the XML tag &lt;code-public-vise&gt;
     * <br>The code contains exactly 5 characters
     * <br>The tag must contains the attribute "ref"
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramTargetAudience(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("ref", "V12"); // The value is fixed for the default implementation
        String[] formacodes = program.getFORMACODE();
        if (formacodes.length > 10)
        {
            getLogger().warn("[" + program.getTitle() + " (" + program.getId() + ")] The LHEO format can have only 10 FORMACODE for XML tag 'code-public-vise'");
        }
        if (formacodes.length == 0)
        {
            getLogger().warn("[" + program.getTitle() + " (" + program.getId() + ")] The LHEO format must have at least one FORMACODE for XML tag 'code-public-vise'");
        }
        
        for (String code : formacodes)
        {
            if (StringUtils.isNotBlank(code))
            {
                _lheoUtils.createLHEOElement(contentHandler, program, "code-public-vise", attrs, code, 5, 5);
            }
        }
    }

    /**
     * Sax the XML tag &lt;niveau-entree-obligatoire&gt;
     * <br>0 for false and 1 for true
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramMandatoryEntryLevel(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        boolean mandatoryEntryLevel = program.isMandatoryEntryLevel();
        String mandatoryEntryLevelAsString = mandatoryEntryLevel ? "1" : "0";
        _lheoUtils.createLHEOElement(contentHandler, program, "niveau-entree-obligatoire", mandatoryEntryLevelAsString);
    }

    /**
     * Sax the XML tag &lt;modalites-alternance&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramApprenticeshipModalities(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String richText2String = _richText2String(program.getApprenticeshipModalities());
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "modalites-alternance", richText2String, 1, 3000);
    }

    /**
     * Sax the XML tag &lt;modalites-enseignement&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProrgamDistanceLearning(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String distanceLearningId = program.getDistanceLearning();
        String distanceLearningForLHEO = null;
        if (StringUtils.isNotBlank(distanceLearningId))
        {
            Content distanceLearningContent = _resolver.resolveById(distanceLearningId);
            
            String code = distanceLearningContent.getValue("code");
            distanceLearningForLHEO = _convertDistanceLearning2LHEO(code);
        }
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "modalites-enseignement", distanceLearningForLHEO);
    }

    /**
     * Convert the ametys code in LHEO code
     * @param code the ametys code
     * @return the LHEO code
     */
    protected String _convertDistanceLearning2LHEO(String code)
    {
        // "A distance"
        if ("distanceLearningModalities_mandatory".equals(code))
        {
            // "formation entièrement à distance"
            return "2";
        }
        // "Hybride"
        else if ("distanceLearningModalities_possible".equals(code))
        {
            // "formation mixte"
            return "1";
        }
        // "En présence"
        else if ("distanceLearningModalities_no".equals(code))
        {
            // "formation entièrement présentielle"
            return "0";
        }
        
        return null;
    }

    /**
     * Sax the XML tag &lt;conditions-specifiques&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramNeededPrerequisite(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String richText2String = _richText2String(program.getNeededPrerequisite());
        _lheoUtils.createMandatoryLHEOElement(contentHandler, program, "conditions-specifiques", richText2String, 1, 3000);
    }

    /**
     * Sax the XML tag &lt;prise-en-charge-frais-possible&gt;
     * <br>0 for false and 1 for true
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramCostBearing(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // In default implementation, the program cost bearing value is always 0 (false)
        _lheoUtils.createLHEOElement(contentHandler, program, "prise-en-charge-frais-possible", "0");
    }

    /**
     * Sax the XML tag &lt;lieu-de-formation&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;coordonnees&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramPlaces(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "lieu-de-formation");
        
        // <coordonnees>
        _saxPlacesCoordonnees(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxPlacesExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "lieu-de-formation");
    }
    
    /**
     * Sax for places the XML tag &lt;coordonnees&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * <br>[0,1] &lt;adresse&gt;
     * <br>[0,1] &lt;telfixe&gt;
     * <br>[0,1] &lt;portable&gt;
     * <br>[0,1] &lt;fax&gt;
     * <br>[0,1] &lt;courriel&gt;
     * <br>[0,1] &lt;web&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxPlacesCoordonnees(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "coordonnees");
        
        List<String> places = Stream.of(program.getPlace())
            .filter(StringUtils::isNotBlank)
            .map(id -> _getRefContent(id))
            .filter(Objects::nonNull)
            .map(c -> c.getTitle())
            .collect(Collectors.toList());
        
        _lheoUtils.createCoordinateLHEOElementsPart1(
                contentHandler, 
                program,
                null, // civilite 
                null, // nom
                null, // prenom
                places // ligne
        );
        
        XMLUtils.endElement(contentHandler, "coordonnees");
    }
    
    /**
     * Sax for places the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxPlacesExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }

    /**
     * Sax the XML tag &lt;modalites-entrees-sorties&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramEntryExitModalities(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // In default implementation, the program entry exit modalities value is always 0 ('entrées/sorties à dates fixes')
        _lheoUtils.createLHEOElement(contentHandler, program, "modalites-entrees-sorties", "1");
    }

    /**
     * Sax the XML tag &lt;session&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;periode&gt;
     * <br>[1,1] &lt;adresse-inscription&gt;
     * <br>[0,1] &lt;modalites-inscription&gt;
     * <br>[0,1] &lt;periode-inscription&gt;
     * <br>[0,1] &lt;etat-recrutement&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramSession(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "session", _getSessionAttributes(program, additionalParameters));
        
        // <periode>
        _saxProgramPeriod(contentHandler, program, additionalParameters);
        
        // <adresse-inscription>
        _saxProgramRegistrationAddress(contentHandler, program, additionalParameters);
        
        // <periode-inscription>
        _saxProgramInscriptionPeriod(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxSessionExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "session");
    }

    /**
     * Get attribute for XML tag &lt;session&gt;
     * @param program the program
     * @param additionalParameters the additional parameters
     * @return the attributes for XML tag &lt;session&gt;
     */
    protected AttributesImpl _getSessionAttributes(AbstractProgram program, Map<String, Object> additionalParameters)
    {
        AttributesImpl attributesImpl = new AttributesImpl();
        attributesImpl.addCDATAAttribute("numero", program.getCode());
        
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        
        ZonedDateTime creationDate = program.getCreationDate();
        LocalDate creationDateToLocalDate = DateUtils.asLocalDate(creationDate.toInstant());
        attributesImpl.addCDATAAttribute("datecrea", formatter.format(creationDateToLocalDate));
        
        ZonedDateTime lastModificationDate = program.getLastModified();
        LocalDate lastModificationDateToLocalDate = DateUtils.asLocalDate(lastModificationDate.toInstant());
        attributesImpl.addCDATAAttribute("datemaj", formatter.format(lastModificationDateToLocalDate));
        
        return attributesImpl;
    }
    
    /**
     * Sax the XML tag &lt;periode&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;debut&gt;
     * <br>[1,1] &lt;fin&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramPeriod(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "periode");
        
        _lheoUtils.createPeriodLHEOElments(contentHandler, program, program.getTeachingStart(), program.getTeachingEnd());
        
        // <extras>
        _saxPeriodExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "periode");
    }

    /**
     * Sax the XML tag &lt;periode-inscription&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;periode&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramInscriptionPeriod(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "periode-inscription");
        
        XMLUtils.startElement(contentHandler, "periode");
        
        _lheoUtils.createPeriodLHEOElments(contentHandler, program, program.getRegistrationStart(), program.getRegistrationDeadline());
        
        _saxProgramInscriptionPeriodExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "periode");
        
        XMLUtils.endElement(contentHandler, "periode-inscription");
    }
    
    /**
     * Sax for period the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxPeriodExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax for inscription period the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramInscriptionPeriodExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }

    /**
     * Sax the XML tag &lt;adresse-inscription&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;adresse&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramRegistrationAddress(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "adresse-inscription");
        
        // <adresse>
        _saxRegistrationAddress(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxRegistrationAddressExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "adresse-inscription");
    }

    /**
     * Sax the XML tag &lt;adresse&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,4] &lt;ligne&gt;
     * <br>[1,1] &lt;codepostal&gt;
     * <br>[1,1] &lt;ville&gt;
     * <br>[0,1] &lt;departement&gt;
     * <br>[0,1] &lt;code-INSEE-commune&gt;
     * <br>[0,1] &lt;code-INSEE-canton&gt;
     * <br>[0,1] &lt;region&gt;
     * <br>[0,1] &lt;pays&gt;
     * <br>[0,1] &lt;geolocalisation&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxRegistrationAddress(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<Person> contactOpt = _getFirstContact(program);
        XMLUtils.startElement(contentHandler, "adresse");
        
        Content contentSaxed = contactOpt.isPresent() ? contactOpt.get() : program;
        String address = contactOpt.isPresent() ? contactOpt.get().getValue("address") : null;
        String zipCode = contactOpt.isPresent() ? contactOpt.get().getValue("zipCode") : null;
        String town = contactOpt.isPresent() ? contactOpt.get().getValue("town") : null;
        
        _lheoUtils.createAddressLHEOElements(
                contentHandler,
                contentSaxed,
                address, // ligne
                zipCode, // codepostal
                town, // ville
                null, // departement
                null, // code-INSEE-commune
                null, // code-INSEE-canton
                null, // region
                null, // pays
                null, // geolocalisation/latitude
                null // geolocalisation/longitude
        );
        XMLUtils.endElement(contentHandler, "adresse");
    }

    /**
     * Sax for registration address the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxRegistrationAddressExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax for action the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxSessionExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;langue-formation&gt;
     * <br>The value contains exactly 2 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramEducationLanguage(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<String> languageOpt = Stream.of(program.getEducationLanguage())
                .filter(StringUtils::isNotBlank)
                .map(id -> _getRefContent(id))
                .filter(Objects::nonNull)
                .map(c -> (String) c.getValue("code"))
                .filter(StringUtils::isNotBlank)
                .findFirst();
                
        _lheoUtils.createLHEOElement(contentHandler, program, "langue-formation", languageOpt.isPresent() ? languageOpt.get() : null, 2, 2);
    }
    
    /**
     * Sax the XML tag &lt;modalites-recrutement&gt;
     * <br>The value contains between 1 to 3000 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramAccessCondition(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        String richText2String = _richText2String(program.getAccessCondition());
        _lheoUtils.createLHEOElement(contentHandler, program, "modalites-recrutement", richText2String, 0, 3000);
    }
    
    /**
     * Sax the XML tag &lt;code-modalite-pedagogique&gt;
     * <br>The code contains exactly 5 characters
     * <br>The tag must contains the attribute "ref"
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramEducationalModalities(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // Nothing in default implementation
    }
    
    /**
     * Sax the XML tag &lt;organisme-formateur&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,4] &lt;SIRET-formateur&gt;
     * <br>[1,1] &lt;raison-sociale-formateur&gt;
     * <br>[1,1] &lt;contact-formateur&gt;
     * <br>[0,1] &lt;potentiel&gt;
     * <br>[0,1] &lt;code-UAI-formateur&gt;
     * <br>[0,1] &lt;reference-certification&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormer(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgunit = _getFirstOrgUnit(program);
        if (orgunit.isPresent())
        {
            XMLUtils.startElement(contentHandler, "organisme-formateur");
            
            // <SIRET-formateur>
            _saxProgramOrgUnitFormerSIRETInformation(contentHandler, program, additionalParameters);
            
            // <raison-sociale-formateur>
            _saxProgramOrgUnitFormerSocialReason(contentHandler, program, additionalParameters);
            
            // <contact-formateur>
            _saxProgramOrgUnitFormerContact(contentHandler, program, additionalParameters);
            
            // <extras>
            _saxProgramOrgUnitFormerExtras(contentHandler, program, additionalParameters);
            
            XMLUtils.endElement(contentHandler, "organisme-formateur");
        }
    }
    
    /**
     * Sax the XML tag &lt;SIRET-formateur&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;SIRET&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerSIRETInformation(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "SIRET-formateur");
        
        // <SIRET>
        _saxProgramOrgUnitFormerSIRET(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxProgramOrgUnitFormerSIRETExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "SIRET-formateur");
    }
    
    /**
     * Sax the XML tag &lt;SIRET&gt;
     * <br>The value contains exactly 14 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerSIRET(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgUnit = _getFirstOrgUnit(program);
        _lheoUtils.createMandatoryLHEOElement(contentHandler, orgUnit.isPresent() ? orgUnit.get() : program, "SIRET", orgUnit.isPresent() ? orgUnit.get().getSIRET() : null, 14, 14); 
    }

    /**
     * Sax for SIRET information the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerSIRETExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;raison-sociale-formateur&gt;
     * <br>The value contains between 1 to 255 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerSocialReason(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgUnit = _getFirstOrgUnit(program);
        _lheoUtils.createMandatoryLHEOElement(contentHandler, orgUnit.isPresent() ? orgUnit.get() : program, "raison-sociale-formateur", orgUnit.isPresent() ? orgUnit.get().getTitle() : null, 1, 255);
    }
    
    /**
     * Sax the XML tag &lt;contact-formateur&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;coordonnees&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerContact(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "contact-formateur");
        
        // <coordonnees>
        _saxProgramOrgUnitFormerContactCoordinates(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxProgramOrgUnitFormerContactExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "contact-formateur");
    }
    
    /**
     * Sax for contact former orgUnit the XML tag &lt;coordonnees&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * <br>[0,1] &lt;adresse&gt;
     * <br>[0,1] &lt;telfixe&gt;
     * <br>[0,1] &lt;portable&gt;
     * <br>[0,1] &lt;fax&gt;
     * <br>[0,1] &lt;courriel&gt;
     * <br>[0,1] &lt;web&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerContactCoordinates(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "coordonnees");
        
        Optional<Person> personOpt = _getFirstOrgUnit(program).stream()
            .map(OrgUnit::getContacts)
            .flatMap(List::stream)
            .filter(StringUtils::isNotBlank)
            .map(this::_getPerson)
            .filter(Objects::nonNull)
            .findFirst();
            
        if (personOpt.isPresent())
        {
            Person person = personOpt.get();
            _saxPersonCoordinate(contentHandler, person, additionalParameters);
        }
        
        XMLUtils.endElement(contentHandler, "coordonnees");
    }
    
    /**
     * Sax for contact former orgUnit the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerContactExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax for former orgUnit the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFormerExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;organisme-financeur&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;code-financeur&gt;
     * <br>[0,1] &lt;nb-places-financees&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitFunder(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // Nothing in default implementation
    }
    
    /**
     * Sax for action the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxActionExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;organisme-formation-responsable&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;numero-activite&gt;
     * <br>[1,1] &lt;SIRET-organisme-formation&gt;
     * <br>[1,1] &lt;nom-organisme&gt;
     * <br>[1,1] &lt;raison-sociale&gt;
     * <br>[1,1] &lt;coordonnees-organisme&gt;
     * <br>[1,1] &lt;contact-organisme&gt;
     * <br>[0,1] &lt;renseignements-specifiques&gt;
     * <br>[0,1] &lt;potentiel&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramResponsibleOrgUnit(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "organisme-formation-responsable", _getResponsibleOrgUnitAttribut(program, additionalParameters));
        
        // <numero-activite>
        _saxProgramActivityNumber(contentHandler, program, additionalParameters);
        
        // <SIRET-organisme-formation>
        _saxProgramSIRETInformation(contentHandler, program, additionalParameters);
        
        // <nom-organisme>
        _saxProgramOrgUnitName(contentHandler, program, additionalParameters);
        
        // <raison-sociale>
        _saxProgramCorporateName(contentHandler, program, additionalParameters);
        
        // <coordonnees-organisme>
        _saxProgramOrgUnitDetails(contentHandler, program, additionalParameters);
        
        // <contact-organisme>
        _saxProgramContactOrgUnit(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxResponsibleOrgUnitExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "organisme-formation-responsable");
    }
    
    /**
     * Get attribute for XML tag &lt;organisme-formation-responsable&gt;
     * @param program the program
     * @param additionalParameters the additional parameters
     * @return the attributes for XML tag &lt;session&gt;
     */
    protected AttributesImpl _getResponsibleOrgUnitAttribut(AbstractProgram program, Map<String, Object> additionalParameters)
    {
        return new AttributesImpl();
    }

    /**
     * Sax the XML tag &lt;numero-activite&gt;
     * <br>The value contains exactly 11 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramActivityNumber(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgUnit = _getFirstOrgUnit(program);
        
        // In default implementation, the program activity number value is always '00000000000'
        String activityNumber = orgUnit
            .map(c -> c.<String>getValue(OrgUnit.ACTIVITY_NUMBER))
            .orElseGet(() -> "00000000000");
        
        _lheoUtils.createMandatoryLHEOElement(contentHandler, orgUnit.isPresent() ? orgUnit.get() : program, "numero-activite", activityNumber, 11, 11);
    }

    /**
     * Sax the XML tag &lt;SIRET-organisme-formation&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;SIRET&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramSIRETInformation(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "SIRET-organisme-formation");
        
        // <SIRET>
        _saxProgramSIRET(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxSIRETInformationExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "SIRET-organisme-formation");
    }

    /**
     * Sax the XML tag &lt;SIRET&gt;
     * <br>The value contains exactly 14 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramSIRET(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgUnit = _getFirstOrgUnit(program);
        
        _lheoUtils.createMandatoryLHEOElement(contentHandler, orgUnit.isPresent() ? orgUnit.get() : program, "SIRET", orgUnit.isPresent() ? orgUnit.get().getSIRET() : null, 14, 14); 
    }

    /**
     * Sax for SIRET information the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxSIRETInformationExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;nom-organisme&gt;
     * <br>The value contains between 1 to 255 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitName(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgUnit = _getFirstOrgUnit(program);
        
        _lheoUtils.createMandatoryLHEOElement(contentHandler, orgUnit.isPresent() ? orgUnit.get() : program, "nom-organisme", orgUnit.isPresent() ? orgUnit.get().getTitle() : null, 1, 255);
    }

    /**
     * Sax the XML tag &lt;raison-sociale&gt;
     * <br>The value contains between 1 to 255 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramCorporateName(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<OrgUnit> orgUnit = _getFirstOrgUnit(program);
        _lheoUtils.createMandatoryLHEOElement(contentHandler, orgUnit.isPresent() ? orgUnit.get() : program, "raison-sociale", orgUnit.isPresent() ? orgUnit.get().getTitle() : null, 1, 255);
    }

    /**
     * Sax the XML tag &lt;coordonnees-organisme&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;coordonnees&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramOrgUnitDetails(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "coordonnees-organisme");
        
        // <coordonnees>
        _saxOrgUnitDetailsCoordinates(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxOrgUnitDetailsExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "coordonnees-organisme");
    }

    /**
     * Sax for orgUnit details the XML tag &lt;coordonnees&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * <br>[0,1] &lt;adresse&gt;
     * <br>[0,1] &lt;telfixe&gt;
     * <br>[0,1] &lt;portable&gt;
     * <br>[0,1] &lt;fax&gt;
     * <br>[0,1] &lt;courriel&gt;
     * <br>[0,1] &lt;web&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxOrgUnitDetailsCoordinates(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "coordonnees");
        
        Optional<Person> personOpt = _getFirstContactFromOrgUnits(program);
        if (personOpt.isPresent())
        {
            Person person = personOpt.get();
            _saxPersonCoordinate(contentHandler, person, additionalParameters);
        }
        
        XMLUtils.endElement(contentHandler, "coordonnees");
    }
    
    /**
     * Sax for orgUnit details the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxOrgUnitDetailsExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;contact-organisme&gt;
     * <br>Can contain the following XML tags:
     * <br>[1,1] &lt;coordonnees&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramContactOrgUnit(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "contact-organisme");
        
        // <coordonnees>
        _saxContactOrgUnitCoordinates(contentHandler, program, additionalParameters);
        
        // <extras>
        _saxContactOrgUnitExtras(contentHandler, program, additionalParameters);
        
        XMLUtils.endElement(contentHandler, "contact-organisme");
    }
    
    /**
     * Sax for contact orgUnit the XML tag &lt;coordonnees&gt;
     * <br>Can contain the following XML tags:
     * <br>[0,1] &lt;civilite&gt;
     * <br>[0,1] &lt;nom&gt;
     * <br>[0,1] &lt;prenom&gt;
     * <br>[0,3] &lt;ligne&gt;
     * <br>[0,1] &lt;adresse&gt;
     * <br>[0,1] &lt;telfixe&gt;
     * <br>[0,1] &lt;portable&gt;
     * <br>[0,1] &lt;fax&gt;
     * <br>[0,1] &lt;courriel&gt;
     * <br>[0,1] &lt;web&gt;
     * <br>[0,N] &lt;extras&gt;
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxContactOrgUnitCoordinates(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "coordonnees");
        
        Optional<Person> personOpt = _getFirstContact(program);
        if (personOpt.isPresent())
        {
            Person person = personOpt.get();
            _saxPersonCoordinate(contentHandler, person, additionalParameters);
        }
        
        XMLUtils.endElement(contentHandler, "coordonnees");
    }
    
    /**
     * Sax for contact orgUnit the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxContactOrgUnitExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax for responsible orgUnit the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxResponsibleOrgUnitExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax the XML tag &lt;code-RNCP&gt;
     * <br>The value contains between 1 and 6 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramRNCPCode(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        Optional<String> rncpCode = Stream.of(program.getRncpCode())
                .filter(StringUtils::isNotBlank)
                .findFirst();
        
        _lheoUtils.createLHEOElement(contentHandler, program, "code-RNCP", rncpCode.isPresent() ? rncpCode.get() : null, 1, 6);
    }
    
    /**
     * Sax the XML tag &lt;code-CERTIFINFO&gt;
     * <br>The value contains between 1 and 6 characters
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramCERTIFINFOCode(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No CERTIFINFO code in default implementation
    }
    
    /**
     * Sax for certification the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxCertificationExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax for programs the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param program the program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxProgramsExtras(ContentHandler contentHandler, AbstractProgram program, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax for offers the XML tag &lt;extras&gt;
     * <br>Can contains all not LHEO normalized elements
     * @param contentHandler the content handler
     * @param programs the list of program to sax
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxOffersExtras(ContentHandler contentHandler, List<AbstractProgram> programs, Map<String, Object> additionalParameters) throws SAXException
    {
        // No extras in default implementation
        XMLUtils.createElement(contentHandler, "extras");
    }
    
    /**
     * Sax LHEO elements of a ametys person for the XML tag &gt;coordonnees&lt;
     * @param contentHandler the content handler
     * @param contact the person contact
     * @param additionalParameters the additional parameters
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxPersonCoordinate(ContentHandler contentHandler, Person contact, Map<String, Object> additionalParameters) throws SAXException
    {
        _lheoUtils.createCoordinateLHEOElementsPart1(
                contentHandler,
                contact,
                contact.getValue("personTitle"), // civilite 
                contact.getValue("lastName"), // nom
                contact.getValue("givenName"), // prenom
                (String) contact.getValue("address") // ligne
        );
        
        XMLUtils.startElement(contentHandler, "adresse");
        _lheoUtils.createAddressLHEOElements(
                contentHandler,
                contact,
                (String) contact.getValue("address"), // ligne
                contact.getValue("zipCode"), // codepostal
                contact.getValue("town"), // ville
                null, // departement
                null, // code-INSEE-commune
                null, // code-INSEE-canton
                null, // region
                null, // pays
                null, // geolocalisation/latitude
                null // geolocalisation/longitude
        );
        XMLUtils.endElement(contentHandler, "adresse");
        
        _lheoUtils.createCoordinateLHEOElementsPart2(
                contentHandler,
                contact,
                contact.getValue("phone"), // telfix
                null, // portable
                contact.getValue("fax"), //fax
                contact.getValue("mail"), // courriel
                contact.getValue("webLinkUrl") // web
        );
    }
    
    /**
     * Get ref content from id
     * @param id the ref content id
     * @return the ref content. Null if no exist
     */
    protected Content _getRefContent(String id)
    {
        try
        {
            return _resolver.resolveById(id);
        }
        catch (Exception e) 
        {
            getLogger().warn("Can't find person with id " + id, e);
            return null;
        }
    }
    
    /**
     * Get first contact from abstract program
     * @param program the abstract program
     * @return the first contact if exist
     */
    protected Optional<Person> _getFirstContact(AbstractProgram program)
    {
        Set<String> contacts = program.getContacts();
        return contacts
                .stream()
                .filter(StringUtils::isNotBlank)
                .map(id -> _getPerson(id))
                .filter(Objects::nonNull)
                .findFirst();
    }
    
    /**
     * Get first contact from orgunits of the abstract program
     * @param program the abstract program
     * @return the first contact if exist
     */
    protected Optional<Person> _getFirstContactFromOrgUnits(AbstractProgram program)
    {
        List<String> orgUnits = program.getOrgUnits();
        return orgUnits
            .stream()
            .filter(StringUtils::isNotBlank)
            .map(id -> _getOrgUnit(id))
            .filter(Objects::nonNull)
            .map(OrgUnit::getContacts)
            .flatMap(List::stream)
            .filter(StringUtils::isNotBlank)
            .map(id -> _getPerson(id))
            .filter(Objects::nonNull)
            .findFirst();
    }
    
    /**
     * Get first orgunit of the abstract program
     * @param program the abstract program
     * @return the first orgunit
     */
    protected Optional<OrgUnit> _getFirstOrgUnit(AbstractProgram program)
    {
        List<String> orgUnits = program.getOrgUnits();
        return orgUnits
                .stream()
                .filter(StringUtils::isNotBlank)
                .map(id -> _getOrgUnit(id))
                .filter(Objects::nonNull)
                .findFirst();
    }
    
    /**
     * Get person from id
     * @param id the person id
     * @return the person content. Null if no exist
     */
    protected Person _getPerson(String id)
    {
        try
        {
            return _resolver.resolveById(id);
        }
        catch (Exception e) 
        {
            getLogger().warn("Can't find person with id " + id, e);
            return null;
        }
    }
    
    /**
     * Get orgUnit from id
     * @param id the orgUnit id
     * @return the orgUnit content. Null if no exist
     */
    protected OrgUnit _getOrgUnit(String id)
    {
        try
        {
            return _resolver.resolveById(id);
        }
        catch (Exception e) 
        {
            getLogger().warn("Can't find orgUnit with id " + id, e);
            return null;
        }
    }
    
    /**
     * Convert rich-text to string for LHEO
     * @param richText the rich-text
     * @return the transformed rich-text
     */
    protected String _richText2String(RichText richText)
    {
        return richText != null ? _richTextHelper.richTextToString(richText) : null;
    }
}
