/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* Imports */
const IOUtils = Java.type("org.apache.commons.io.IOUtils");
const FileUtils = Java.type("org.apache.commons.io.FileUtils");

/* Components */
const xPathProcessor = Ametys.serviceManager.lookup(org.apache.excalibur.xml.xpath.XPathProcessor.ROLE);
const domParser = Ametys.serviceManager.lookup(org.apache.excalibur.xml.dom.DOMParser.ROLE);

function _setMentionType(mentionType, content)
{
    content.setValue("mentionType", mentionType);
}

function _addMentionType(configNode, configIdForDegree, mentionType)
{
    let degreeCode = xPathProcessor.evaluateAsString(configNode, configIdForDegree);
    if (degreeCode == null)
    {
        logger.warn("No degree code found associate to '" + configIdForDegree + "' in the general configuration");
        return;
    }
    
    let count = 0;
    Repository.query("//element(*, ametys:content)[@ametys-internal:contentType = 'odf-enumeration.Degree' and @ametys:code = '" + degreeCode + "']") 
        .forEach(function(content) 
        { 
            Content.migrate( 
                content, 
                [_setMentionType.bind(null, mentionType)], 
                true, /* old version still comptatible */ 
                null, /* no tag */ 
                false, /* verbose */
                true /* synchronize live */
            );
            count++;
            logger.info("The degree content '" + content.getTitle() + "' (" + content.getId() + ") with code " + degreeCode + " was linked to mention type " + mentionType);
        } 
    ); 
    logger.info(count + " degree(s) with code " + degreeCode + " has been linked to mention type " + mentionType);
}

function _removeObsoleteConfigValues()
{
    let configValues = org.ametys.runtime.config.Config.getInstance().getValues();
    let currentConfigFile = new java.io.File(configFolder, configFilename);
    org.ametys.runtime.config.ConfigManager.getInstance().save(configValues, currentConfigFile.getAbsolutePath());
}

let configFolder = org.ametys.runtime.util.AmetysHomeHelper.getAmetysHomeConfig();
let configFilename = org.ametys.runtime.servlet.RuntimeServlet.CONFIG_FILE_NAME;
let configFile = new java.io.File(configFolder, configFilename + ".ODF-2790");
if (!configFile.exists())
{
    logger.info("The file '" + configFile.getName() + "' doesn't exist. The migration is useless.");
}
else
{
    let fis = new java.io.FileInputStream(configFile);
    try
    {
        let doc = domParser.parseDocument(new org.xml.sax.InputSource(fis));
        let configNode = doc.getFirstChild();
        
        _addMentionType(configNode, "odf.programs.degree.license", "odf-enumeration.MentionLicence");
        _addMentionType(configNode, "odf.programs.degree.licensepro", "odf-enumeration.MentionLicencepro");
        _addMentionType(configNode, "odf.programs.degree.master", "odf-enumeration.MentionMaster");
        
        FileUtils.deleteQuietly(configFile);
        
        _removeObsoleteConfigValues();
    }
    catch (e)
    {
        logger.error("An error has occured during migrating mentions.", e);
        throw e;
    }
    finally
    {
        IOUtils.closeQuietly(fis);
    }
}
