/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * CDM-fr handler to deposit or send CDM-fr
 */
public class CDMFRHandler implements Component, Serviceable, Contextualizable
{
    /** The component role. */
    public static final String ROLE = CDMFRHandler.class.getName();
 
    /** Key to get if the CDM-fr observer is suspended */
    public static final String SUSPEND_CDMFR = "suspend-cdmfr";

    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The send CDM-fr processor */
    protected SendCDMfrProcessor _sendCDMfrProcessor;
    
    /** The deposit CDM-fr processor */
    protected DepositCDMfrProcessor _depositCDMfrProcessor;

    /** The context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _sendCDMfrProcessor = (SendCDMfrProcessor) manager.lookup(SendCDMfrProcessor.ROLE);
        _depositCDMfrProcessor = (DepositCDMfrProcessor) manager.lookup(DepositCDMfrProcessor.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Check if at least one of the external CDM-fr actions is active.
     * @return <code>true</code> if one of the action is active and the handler is useful
     */
    public boolean needToHandle()
    {
        return _sendCDMfrProcessor.isActive() || _depositCDMfrProcessor.isActive();
    }
    
    /**
     * Handle CDM-fr for the given contents
     * @param contents the list of contents
     */
    public void handleCDMFR(List<ModifiableContent> contents)
    {
        if (needToHandle())
        {
            _handleCDMFR(contents);
        }
    }
    
    private void _handleCDMFR(List<ModifiableContent> contents)
    {
        Set<Program> programs = contents.stream()
                .map(this::getProgramParents)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
        
        _sendCDMfrProcessor.processPrograms(programs);
        _depositCDMfrProcessor.processPrograms(programs);
    }
    
    /**
     * Get the program parents for CDM-fr treatment
     * @param content the content
     * @return the set of program parents
     */
    public Set<Program> getProgramParents(Content content)
    {
        return switch (content)
        {
            case OrgUnit orgunit -> _odfHelper.getProgramsReferencingOrgunit(orgunit);
            case Person person -> _odfHelper.getProgramsReferencingPerson(person);
            case ProgramItem programItem -> _odfHelper.getParentPrograms(programItem, true);
            default -> Set.of();
        };
    }
    
    /**
     * <code>true</code> if the CDM-fr observer is suspended
     * @return <code>true</code> if the CDM-fr observer is suspended
     */
    public boolean isSuspended()
    {
        try
        {
            Request request = ContextHelper.getRequest(_context);
            // Check if the suspend date is defined in request attributes
            return Optional.ofNullable(request.getAttribute(SUSPEND_CDMFR)).isPresent();
        }
        catch (Exception e)
        {
            // Exception can occurred if the context is not defined.
            // In this case, we consider that the CDM-fr handler is not suspended,
            // it is better to send to much than not enough
            return false;
        }
    }
    
    /**
     * Suspend CDM-fr observers. If you call this method, the {@link #unsuspendCDMFRObserver(Set)}
     * method should be call at the end of you treatments.
     */
    public void suspendCDMFRObserver()
    {
        // The method should not be call twice until the unsuspend method has not been called.
        if (isSuspended())
        {
            throw new UnsupportedOperationException("The CDM-fr observers are already suspended, this method should not be called until the CDM-fr observers are unsuspended.");
        }
        
        Request request = ContextHelper.getRequest(_context);
        // Add the suspend date from request attributes
        request.setAttribute(SUSPEND_CDMFR, ZonedDateTime.now());
    }
    
    /**
     * Unsuspend CDM-fr observers. You can give content identifiers to the method to be handled.
     * @param waitingContentIds The collection of content identifiers waiting to be handle, can be null
     */
    public void unsuspendCDMFRObserver(Set<String> waitingContentIds)
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Do not do useless operations if actions are inactives
        if (needToHandle() && waitingContentIds != null && isSuspended())
        {
            // Get the suspend date from request attributes
            ZonedDateTime suspendDate = (ZonedDateTime) request.getAttribute(SUSPEND_CDMFR);
            
            // Get the validated contents into the collection from the begin to now to handle them
            List<ModifiableContent> validatedContents = waitingContentIds.stream()
                    .map(_resolver::<ModifiableContent>resolveById)
                    .filter(c -> _hasBeenValidatedAfter(c, suspendDate))
                    .toList();
            
            _handleCDMFR(validatedContents);
        }
        
        // Remove the suspend date from request attributes
        request.removeAttribute(SUSPEND_CDMFR);
    }
    
    private boolean _hasBeenValidatedAfter(Content content, ZonedDateTime suspendDate)
    {
        return Optional.of(content)
            .map(Content::getLastValidationDate)
            .map(date -> date.isAfter(suspendDate))
            .orElse(false);
    }
}
