/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import org.ametys.odf.program.Program;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Processor to deposit the CDM-fr file
 */
public class DepositCDMfrProcessor extends AbstractCDMfrProcessor
{
    /** The component role. */
    public static final String ROLE = DepositCDMfrProcessor.class.getName();
    
    private File _outputFolder;
    
    @Override
    public void initialize() throws Exception
    {
        super.initialize();
        if (isActive())
        {
            _outputFolder = new File(AmetysHomeHelper.getAmetysHomeData(), "/odf/cdmfr");
            FileUtils.forceMkdir(_outputFolder);
        }
    }
    @Override
    protected String getConfigActiveParam()
    {
        return "odf.publish.cdm-fr.output.folder";
    }
    
    @Override
    protected void processProgram(Program program, InputStream cdmfrContent) throws IOException
    {
        String filename = program.getCDMId() + ".xml";
        
        // Delete existing file
        File file = new File(_outputFolder, program.getCatalog() + File.separator + filename);
        FileUtils.forceMkdirParent(file);
        if (file.exists())
        {
            file.delete();
        }

        // Deposit CDM-fr for the given programs
        try (OutputStream os = new FileOutputStream(file))
        {
            IOUtils.copy(cdmfrContent, os);
        }
    }
    
    @Override
    protected boolean isCDMfrForAmetys()
    {
        return false;
    }
    
    @Override
    protected String getErrorMessage()
    {
        return "Unable to generate and copy the CDM-fr file";
    }
}
