/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation.skill;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.data.ContentValue;

/**
 * Abstract observer to prepare skills step observer when a content is updated
 */
public abstract class AbstractSkillsStepObserver extends AbstractSkillsObserver implements Contextualizable
{
    /** The context */
    protected Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the skills to delete
     * @param previousSkills The previous skills
     * @param currentSkills The current skills
     * @return The skills to delete
     */
    protected ContentValue[] getSkillsToDelete(ContentValue[] previousSkills, ContentValue[] currentSkills)
    {
        // If the current skills are null or empty, we need to delete the previous skills
        ContentValue[] skillsToDelete = previousSkills;
        // If the current skills are not null or empty, we need to compare them with the previous skills to retrieve the ones that need to be deleted
        if (previousSkills != null && currentSkills != null)
        {
            // Get the skills that were in the skills but are not after modification
            for (ContentValue newValue : currentSkills)
            {
                skillsToDelete = ArrayUtils.removeElement(skillsToDelete, newValue);
            }
        }
        
        return skillsToDelete;
    }
    
    /**
     * Get request attribute for a content
     * @param <T> The type of results
     * @param attributeName The name of the attribute
     * @param contentId The content identifier
     * @return The value found in request or default value
     */
    protected <T> T _getRequestAttribute(String attributeName, String contentId)
    {
        return this.<T>_getRequestAttribute(ContextHelper.getRequest(_context), attributeName)
            .get(contentId);
    }
    
    @SuppressWarnings("unchecked")
    private <T> Map<String, T> _getRequestAttribute(Request request, String attributeName)
    {
        return Optional.of(request)
            .map(r -> (Map<String, T>) r.getAttribute(attributeName))
            .orElseGet(HashMap::new);
    }
    
    /**
     * Set a request attribute for a content
     * @param <T> The type of the attribute
     * @param attributeName The name of the attribute
     * @param contentId The content identifier
     * @param value The value
     */
    protected <T> void _setRequestAttribute(String attributeName, String contentId, T value)
    {
        Request request = ContextHelper.getRequest(_context);
        Map<String, T> attribute = _getRequestAttribute(request, attributeName);
        attribute.put(contentId, value);
        request.setAttribute(attributeName, attribute);
    }
}
