/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.model.ContentElementDefinition;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.provider.WorkspaceSelector;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.google.common.collect.Maps;

/**
 * Component providing methods to retrieve ODF virtual pages, such as the ODF root,
 * level 1 and 2 metadata names, and so on.
 */
public class OdfClassificationHandler extends AbstractLogEnabled implements Component, Initializable, Serviceable
{
    /** The avalon role. */
    public static final String ROLE = OdfClassificationHandler.class.getName();
    
    /** First level attribute name. */
    public static final String LEVEL1_ATTRIBUTE_NAME = "firstLevel";
    
    /** Second level attribute name. */
    public static final String LEVEL2_ATTRIBUTE_NAME = "secondLevel";
    
    /** Catalog attribute name. */
    public static final String CATALOG_ATTRIBUTE_NAME = "odf-root-catalog";
    
    /** Constant for the {@link Cache} id for the {@link LevelValue}s objects in cache by {@link LevelValuesCacheKey}. */
    public static final String LEVEL_VALUES_CACHE = OdfClassificationHandler.class.getName() + "$LevelValues";
    
    /** The default level 1 attribute. */
    protected static final String DEFAULT_LEVEL1_ATTRIBUTE = "degree";
    
    /** The default level 2 attribute. */
    protected static final String DEFAULT_LEVEL2_ATTRIBUTE = "domain";
    
    /** Content types that are not eligible for first and second level */
    // See ODF-1115 Exclude the mentions enumerator from the list :
    protected static final List<String> NON_ELIGIBLE_CTYPES_FOR_LEVEL = Arrays.asList("org.ametys.plugins.odf.Content.programItem", "odf-enumeration.Mention");
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;
    
    /** The content type extension point. */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The ODF Catalog enumeration */
    protected CatalogsManager _catalogsManager;
    
    /** The workspace selector. */
    protected WorkspaceSelector _workspaceSelector;
    
    /** Avalon service manager */
    protected ServiceManager _manager;
    
    /** Content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** Odf reference table helper */
    protected OdfReferenceTableHelper _odfReferenceTableHelper;
    
    /** Root orgunit provider */
    protected RootOrgUnitProvider _orgUnitProvider;
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _manager = serviceManager;
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _workspaceSelector = (WorkspaceSelector) serviceManager.lookup(WorkspaceSelector.ROLE);
        _catalogsManager = (CatalogsManager) serviceManager.lookup(CatalogsManager.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _odfReferenceTableHelper = (OdfReferenceTableHelper) serviceManager.lookup(OdfReferenceTableHelper.ROLE);
        _orgUnitProvider = (RootOrgUnitProvider) serviceManager.lookup(RootOrgUnitProvider.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(LEVEL_VALUES_CACHE,
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_LEVEL_VALUES_LABEL"),
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_LEVEL_VALUES_DESC"),
                true,
                null);
    }
    
    /**
     * Get the ODF catalogs
     * @return the ODF catalogs
     */
    public Map<String, I18nizableText> getCatalogs ()
    {
        Map<String, I18nizableText> catalogs = new HashMap<>();
        
        for (Catalog catalog : _catalogsManager.getCatalogs())
        {
            catalogs.put(catalog.getName(), new I18nizableText(catalog.getTitle()));
        }
        
        return catalogs;
    }
    
    /**
     * True if the program attribute is eligible
     * @param attributePath the attribute path
     * @param allowMultiple true is we allow multiple attribute
     * @return true if the program attribute is eligible
     */
    public boolean isEligibleMetadataForLevel(String attributePath, boolean allowMultiple)
    {
        ContentType cType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
        
        if (cType.hasModelItem(attributePath))
        {
            ModelItem modelItem = cType.getModelItem(attributePath);
            return _isModelItemEligible(modelItem, allowMultiple);
        }
        else
        {
            return false;
        }
    }
    
    /**
     * Get the eligible enumerated attributes for ODF page level
     * @return the eligible attributes
     */
    public Map<String, ModelItem> getEligibleAttributesForLevel()
    {
        return getEnumeratedAttributes(ProgramFactory.PROGRAM_CONTENT_TYPE, false);
    }
    
    /**
     * Get the enumerated attribute definitions for the given content type.
     * Attribute with enumerator or content attribute are considered as enumerated
     * @param programContentTypeId The content type's id
     * @param allowMultiple <code>true</code> true to allow multiple attribute
     * @return The definitions of enumerated attributes
     */
    public Map<String, ModelItem> getEnumeratedAttributes(String programContentTypeId, boolean allowMultiple)
    {
        ContentType cType = _cTypeEP.getExtension(programContentTypeId);
        
        return _collectEligibleChildAttribute(cType, allowMultiple)
                    .collect(LambdaUtils.Collectors.toLinkedHashMap(Map.Entry::getKey, Map.Entry::getValue));
    }
    
    private Stream<Map.Entry<String, ModelItem>> _collectEligibleChildAttribute(ModelItemContainer modelItemContainer, boolean allowMultiple)
    {
        // repeaters are not supported
        if (modelItemContainer instanceof RepeaterDefinition)
        {
            return Stream.empty();
        }
        
        return modelItemContainer.getModelItems().stream()
            .flatMap(modelItem ->
            {
                if (_isModelItemEligible(modelItem, allowMultiple))
                {
                    return Stream.of(Maps.immutableEntry(modelItem.getPath(), modelItem));
                }
                else if (modelItem instanceof ModelItemContainer)
                {
                    return _collectEligibleChildAttribute((ModelItemContainer) modelItem, allowMultiple);
                }
                else
                {
                    return Stream.empty();
                }
            });
    }
    
    @SuppressWarnings("static-access")
    private boolean _isModelItemEligible(ModelItem modelItem, boolean allowMultiple)
    {
        if (!(modelItem instanceof ElementDefinition))
        {
            return false;
        }
        
        ElementDefinition elementDefinition = (ElementDefinition) modelItem;
        if (elementDefinition.isMultiple() && !allowMultiple)
        {
            return false;
        }
        
        
        String typeId = elementDefinition.getType().getId();
        if (elementDefinition instanceof ContentElementDefinition contentElementDefinition)
        {
            String contentTypeId = contentElementDefinition.getContentTypeId();
            
            Stream<String> selfAndAncestors = Stream.concat(
                Stream.of(contentTypeId),
                _contentTypesHelper.getAncestors(contentTypeId).stream()
            );
            
            return selfAndAncestors.noneMatch(NON_ELIGIBLE_CTYPES_FOR_LEVEL::contains);
        }
        else if (ModelItemTypeConstants.STRING_TYPE_ID.equals(typeId))
        {
            return elementDefinition.getEnumerator() != null && !elementDefinition.getName().startsWith("dc_");
        }
        else
        {
            return false;
        }
    }
    
    /**
     * Get the level value of a program by extracting and transforming the raw program value at the desired metadata path
     * @param program The program
     * @param levelMetaPath The desired metadata path that represent a level
     * @return The list of final level values
     */
    public List<String> getProgramLevelValues(Program program, String levelMetaPath)
    {
        List<String> rawValues = getProgramLevelRawValues(program, levelMetaPath);
        return rawValues.stream()
                .map(e -> _convertRawValue2LevelValue(levelMetaPath, e))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }
    
    /**
     * Get the level value of a program by extracting and transforming the raw program value at the desired attribute path
     * @param program The program
     * @param levelAttributePath The desired attribute path that represent a level
     * @return The list of level raw value
     */
    public List<String> getProgramLevelRawValues(Program program, String levelAttributePath)
    {
        String typeId = program.getType(levelAttributePath).getId();
    
        if (ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(typeId))
        {
            if (program.isMultiple(levelAttributePath))
            {
                return ContentDataHelper.getContentIdsListFromMultipleContentData(program, levelAttributePath);
            }
            else
            {
                return Collections.singletonList(ContentDataHelper.getContentIdFromContentData(program, levelAttributePath));
            }
        }
        else if (org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID.equals(typeId))
        {
            if (program.isMultiple(levelAttributePath))
            {
                return Arrays.asList(program.getValue(levelAttributePath));
            }
            else
            {
                return Collections.singletonList(program.getValue(levelAttributePath));
            }
        }
        else
        {
            throw new IllegalArgumentException("The attribute at path '" + levelAttributePath + "' is not an eligible attribute for level");
        }
    }
    
    /**
     * Convert the attribute raw value into a level value
     * @param attributePath The path of the attribute corresponding to the level
     * @param rawLevelValue The raw level value
     * @return the converted value or <code>null</code> if there is no level value for this raw value
     */
    protected String _convertRawValue2LevelValue(String attributePath, String rawLevelValue)
    {
        // FIXME a raw <=> level value cache would be useful, but need a specific cache management strategy
        
        String levelValue = rawLevelValue;
        
        ContentType programCType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
        ModelItem modelItem = programCType.getModelItem(attributePath);
        
        String attributeContentTypeId = null;
        if (modelItem instanceof ContentAttributeDefinition)
        {
            attributeContentTypeId = ((ContentAttributeDefinition) modelItem).getContentTypeId();
        }
        
        if (StringUtils.isNotEmpty(attributeContentTypeId))
        {
            // Odf reference table
            if (_odfReferenceTableHelper.isTableReference(attributeContentTypeId))
            {
                levelValue = _convertRaw2LevelForRefTable(rawLevelValue);
            }
            // Orgunit
            else if (OrgUnitFactory.ORGUNIT_CONTENT_TYPE.equals(attributeContentTypeId))
            {
                levelValue = _convertRaw2LevelForOrgUnit(rawLevelValue);
            }
            // Other content
            else
            {
                levelValue = _convertRaw2LevelForContent(rawLevelValue);
            }
        }
        
        return StringUtils.defaultIfEmpty(levelValue, null);
    }

    private String _convertRaw2LevelForRefTable(String contentId)
    {
        return _odfReferenceTableHelper.getItemCode(contentId);
    }
    
    private String _convertRaw2LevelForOrgUnit(String orgUnitId)
    {
        try
        {
            OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
            return orgUnit.getUAICode();
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Unable to get level value for orgunit with id '{}'.", orgUnitId, e);
            return "";
        }
    }
    
    private String _convertRaw2LevelForContent(String contentId)
    {
        try
        {
//            Content content = _resolver.resolveById(contentId);
            // FIXME name might not be unique between sites, languages, content without site etc...
            // return content.getName();
            return contentId;
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Unable to get level value for content with id '{}'.", contentId, e);
            return "";
        }
    }
    
    private String _convertLevel2RawForRefTable(String metaContentType, String levelValue)
    {
        return Optional.ofNullable(_odfReferenceTableHelper.getItemFromCode(metaContentType, levelValue))
                .map(OdfReferenceTableEntry::getId)
                .orElse(null);
    }
    
    private String _convertLevel2RawForContent(String levelValue)
    {
        // must return the content id
        // FIXME currently the level value is the content id (see #_convertRaw2LevelForContent)
        return levelValue;
    }
    
    /**
     * Clear the cache of available values for levels used for ODF virtual pages
     */
    public void clearLevelValues()
    {
        _getLevelValuesCache().invalidateAll();
    }
    
    /**
     * Clear the cache  of available values for level
     * @param lang the language. Can be null to clear values for all languages
     * @param metadataPath the path of level's metadata
     */
    public void clearLevelValues(String metadataPath, String lang)
    {
        Cache<LevelValuesCacheKey, Map<String, LevelValue>> levelValuesCache = _getLevelValuesCache();
        
        LevelValuesCacheKey levelValuesKey = LevelValuesCacheKey.of(metadataPath, lang, null);
        levelValuesCache.invalidate(levelValuesKey);
    }

    /**
     * Get the first level metadata values (with translated label).
     * @param metadata Metadata of first level
     * @param language Lang to get
     * @return the first level metadata values.
     */
    public Map<String, LevelValue> getLevelValues(String metadata, String language)
    {
        Cache<LevelValuesCacheKey, Map<String, LevelValue>> levelValuesCache = _getLevelValuesCache();
        
        String workspace = _workspaceSelector.getWorkspace();
        LevelValuesCacheKey levelValuesKey = LevelValuesCacheKey.of(metadata, language, workspace);
        
        return levelValuesCache.get(levelValuesKey, __ -> _getLevelValues(metadata, language));
    }
    
    /**
     * Get the available values of a program's attribute to be used as a level in the virtual ODF page hierarchy.
     * @param attributePath the attribute path.
     * @param language the language.
     * @return the available attribute values.
     */
    private Map<String, LevelValue> _getLevelValues(String attributePath, String language)
    {
        try
        {
            ContentType programCType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
            ModelItem modelItem = programCType.getModelItem(attributePath);
            
            String attributeContentTypeId = null;
            if (modelItem instanceof ContentAttributeDefinition)
            {
                attributeContentTypeId = ((ContentAttributeDefinition) modelItem).getContentTypeId();
            }
            
            if (StringUtils.isNotEmpty(attributeContentTypeId))
            {
                // Odf reference table
                if (_odfReferenceTableHelper.isTableReference(attributeContentTypeId))
                {
                    return _getLevelValuesForRefTable(attributeContentTypeId, language);
                }
                // Orgunit
                else if (OrgUnitFactory.ORGUNIT_CONTENT_TYPE.equals(attributeContentTypeId))
                {
                    return _getLevelValuesForOrgUnits();
                }
                // Other content
                else
                {
                    return _getLevelValuesForContentType(attributeContentTypeId, language);
                }
            }
            
            Enumerator<?> enumerator = null;
            if (modelItem instanceof ElementDefinition)
            {
                enumerator = ((ElementDefinition) modelItem).getEnumerator();
            }
            
            if (enumerator != null)
            {
                return _getLevelValuesForEnumerator(language, enumerator);
            }
        }
        catch (Exception e)
        {
            // Log and return empty map.
            getLogger().error("Error retrieving values for metadata {} in language {}", attributePath, language, e);
        }
        
        return Maps.newHashMap();
    }
    
    private Map<String, LevelValue> _getLevelValuesForRefTable(String metaContentType, String language)
    {
        Map<String, LevelValue> levelValues = new LinkedHashMap<>();
        
        List<OdfReferenceTableEntry> entries = _odfReferenceTableHelper.getItems(metaContentType);
        for (OdfReferenceTableEntry entry : entries)
        {
            if (StringUtils.isEmpty(entry.getCode()))
            {
                getLogger().warn("There is no code for entry {} ({}) of reference table '{}'. It will be ignored for classification", entry.getLabel(language), entry.getId(), metaContentType);
            }
            else if (levelValues.containsKey(entry.getCode()))
            {
                getLogger().warn("Duplicate key code {} into reference table '{}'. The entry {} ({}) will be ignored for classification", entry.getCode(), metaContentType, entry.getLabel(language), entry.getId());
            }
            else
            {
                LevelValue levelValue = new LevelValue(
                                            entry.getLabel(language),
                                            entry.getOrder()
                                        );
                levelValues.put(entry.getCode(), levelValue);
            }
        }
        
        return levelValues;
    }
    
    private Map<String, LevelValue> _getLevelValuesForOrgUnits()
    {
        String rootOrgUnitId = _orgUnitProvider.getRootId();
        Set<String> childOrgUnitIds = _orgUnitProvider.getChildOrgUnitIds(rootOrgUnitId, true);
        
        Map<String, LevelValue> levelValues = new LinkedHashMap<>();
        
        for (String childOUId : childOrgUnitIds)
        {
            OrgUnit childOU = _resolver.resolveById(childOUId);
            if (StringUtils.isEmpty(childOU.getUAICode()))
            {
                getLogger().warn("There is no UAI code for orgunit {} ({}). It will be ignored for classification", childOU.getTitle(), childOU.getId());
            }
            else if (levelValues.containsKey(childOU.getUAICode()))
            {
                getLogger().warn("Duplicate UAI code {}. The orgunit {} ({}) will be ignored for classification", childOU.getUAICode(), childOU.getTitle(), childOU.getId());
            }
            else
            {
                levelValues.put(childOU.getUAICode(), _convertToLevelValue(childOU.getTitle()));
            }
        }
        return levelValues;
    }
    
    private Map<String, LevelValue> _getLevelValuesForContentType(String metaContentType, String language)
    {
        Expression expr = new AndExpression(
            new ContentTypeExpression(Operator.EQ, metaContentType),
            new LanguageExpression(Operator.EQ, language)
        );
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr);
        
        return _resolver.<Content>query(xpathQuery).stream()
            .collect(LambdaUtils.Collectors.toLinkedHashMap(Content::getId, c -> _convertToLevelValue(c.getTitle())));
    }
    
    private <T extends Object> Map<String, LevelValue> _getLevelValuesForEnumerator(String language,  Enumerator<T> enumerator) throws Exception
    {
        return enumerator.getEntries().entrySet().stream()
            .filter(entry -> StringUtils.isNotEmpty(entry.getKey().toString()))
            .map(entry ->
            {
                String code = entry.getKey().toString();
                
                I18nizableText label = entry.getValue();
                String itemLabel = _i18nUtils.translate(label, language);
                
                return Maps.immutableEntry(code, itemLabel);
            })
            .collect(LambdaUtils.Collectors.toLinkedHashMap(Map.Entry::getKey, e -> _convertToLevelValue(e.getValue())));
    }
    
    /**
     * Get a collection of programs corresponding to following parameters.
     * @param catalog Name of the catalog. Can be null to get all programs matching other arguments.
     * @param lang the content language. Can not be null.
     * @param level1MetaPath Having a non-empty value for the metadata path
     * @param level1 If this parameter is not null or empty and level1MetaPath too, we filter programs by the metadata level1MetaPath value of level1
     * @param level2MetaPath Having a non-empty value for the metadata path
     * @param level2 If this parameter is not null or empty and level2MetaPath too, we filter programs by the metadata level2MetaPath value of level2
     * @param programCode The program's code. Can be null to get all programs matching other arguments.
     * @param programName The program's name. Can be null to get all programs matching other arguments.
     * @param additionalExpressions Additional expressions to add to search
     * @return A collection of programs
     */
    public AmetysObjectIterable<Program> getPrograms(String catalog, String lang, String level1MetaPath, String level1, String level2MetaPath, String level2, String programCode, String programName, Collection<Expression> additionalExpressions)
    {
        List<Expression> exprs = new ArrayList<>();

        exprs.add(new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE));
        exprs.add(new LanguageExpression(Operator.EQ, lang));

        /* Level 1 */
        if (StringUtils.isNotEmpty(level1))
        {
            exprs.add(new StringExpression(level1MetaPath, Operator.EQ, _convertLevelValue2RawValue(lang, level1MetaPath, level1)));
        }
        else if (StringUtils.isNotBlank(level1MetaPath))
        {
            exprs.add(new StringExpression(level1MetaPath, Operator.NE, ""));
        }

        /* Level 2 */
        if (StringUtils.isNotEmpty(level2))
        {
            exprs.add(new StringExpression(level2MetaPath, Operator.EQ, _convertLevelValue2RawValue(lang, level2MetaPath, level2)));
        }
        else if (StringUtils.isNotBlank(level2MetaPath))
        {
            exprs.add(new StringExpression(level2MetaPath, Operator.NE, ""));
        }
        
        if (catalog != null)
        {
            exprs.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        }
        
        if (StringUtils.isNotEmpty(programCode))
        {
            exprs.add(new StringExpression(ProgramItem.CODE, Operator.EQ, programCode));
        }
        
        if (additionalExpressions != null)
        {
            exprs.addAll(additionalExpressions);
        }
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        Expression contentExpression = new AndExpression(exprs.toArray(new Expression[exprs.size()]));
        
        String xPathQuery = QueryHelper.getXPathQuery(StringUtils.defaultIfEmpty(programName, null), "ametys:content", contentExpression, sortCriteria);
        return _resolver.query(xPathQuery);
    }

    /**
     * Get the orgunit identifier given an uai code
     * @param lang Language
     * @param uaiCode The uai code
     * @return The orgunit id or null if not found
     */
    public String getOrgunitIdFromUaiCode(String lang, String uaiCode)
    {
        Expression ouExpression = new AndExpression(
            new ContentTypeExpression(Operator.EQ, OrgUnitFactory.ORGUNIT_CONTENT_TYPE),
            new LanguageExpression(Operator.EQ, lang),
            new StringExpression(OrgUnit.CODE_UAI, Operator.EQ, uaiCode)
        );
        
        String query = ContentQueryHelper.getContentXPathQuery(ouExpression);
        return _resolver.query(query).stream().findFirst().map(AmetysObject::getId).orElse(null);
    }

    /**
     * Convert a level value to the raw value
     * @param lang The language
     * @param levelAttribute The name of attribute holding the level
     * @param levelValue The level value
     * @return The raw value
     */
    private String _convertLevelValue2RawValue(String lang, String levelAttribute, String levelValue)
    {
        // FIXME a raw <=> level value cache would be useful, but need a specific cache management strategy
        
        String rawValue = null;
        
        ContentType programCType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
        ModelItem modelItem = programCType.getModelItem(levelAttribute);
        
        String attributeContentTypeId = null;
        if (modelItem instanceof ContentAttributeDefinition)
        {
            attributeContentTypeId = ((ContentAttributeDefinition) modelItem).getContentTypeId();
        }
        
        if (StringUtils.isNotEmpty(attributeContentTypeId))
        {
            if (_odfReferenceTableHelper.isTableReference(attributeContentTypeId))
            {
                rawValue = _convertLevel2RawForRefTable(attributeContentTypeId, levelValue);
            }
            // Orgunit
            else if (OrgUnitFactory.ORGUNIT_CONTENT_TYPE.equals(attributeContentTypeId))
            {
                rawValue = _convertLevel2RawForOrgUnit(lang, levelValue);
            }
            // Other content
            else
            {
                rawValue = _convertLevel2RawForContent(levelValue);
            }
        }
        
        return StringUtils.defaultIfEmpty(rawValue, levelValue);
    }
    
    private String _convertLevel2RawForOrgUnit(String lang, String levelValue)
    {
        return getOrgunitIdFromUaiCode(lang, levelValue);
    }
    
    /**
     * Organize passed programs by levels into a Map.
     * @param programs Programs to organize
     * @param level1 Name of the metadata of first level
     * @param level2 Name of the metadata of second level
     * @return A Map of Map with a Collection of programs which representing the organization of programs by levels.
     * @throws SAXException if an error occured
     */
    public Map<String, Map<String, Collection<Program>>> organizeProgramsByLevels(AmetysObjectIterable<Program> programs, String level1, String level2) throws SAXException
    {
        Map<String, Map<String, Collection<Program>>> level1Map = new TreeMap<>();
        
        for (Program program : programs)
        {
            List<String> programL1RawValues = getProgramLevelRawValues(program, level1);
            List<String> programL2RawValues = getProgramLevelRawValues(program, level2);
            for (String programL1Value : programL1RawValues)
            {
                if (StringUtils.isNotEmpty(programL1Value))
                {
                    Map<String, Collection<Program>> level2Map = level1Map.computeIfAbsent(programL1Value, x -> new TreeMap<>());
                    for (String programL2Value : programL2RawValues)
                    {
                        if (StringUtils.isNotEmpty(programL2Value))
                        {
                            Collection<Program> programCache = level2Map.computeIfAbsent(programL2Value, x -> new ArrayList<>());
                            programCache.add(program);
                        }
                    }
                }
            }
        }
        
        return level1Map;
    }
    
    private LevelValue _convertToLevelValue(String value)
    {
        return new LevelValue(value, Long.MAX_VALUE);
    }
    
    /**
     * Wrapper object for a level value
     */
    public static class LevelValue
    {
        private String _value;
        private Long _order;
        
        /**
         * The constructor
         * @param value the value
         * @param order the order
         */
        public LevelValue(String value, Long order)
        {
            _value = value;
            _order = order;
        }
        
        /**
         * Get the value
         * @return the value
         */
        public String getValue()
        {
            return _value;
        }
        
        /**
         * Get the order
         * @return the order
         */
        public Long getOrder()
        {
            return _order;
        }
        
        /**
         * Compare to a level value depends of the order first then the value
         * @param levelValue the level value to compare
         * @return the int value of the comparaison
         */
        public int compareTo(LevelValue levelValue)
        {
            if (_order.equals(levelValue.getOrder()))
            {
                String value1 = Normalizer.normalize(_value, Normalizer.Form.NFD).replaceAll("[\\p{InCombiningDiacriticalMarks}]", "");
                String value2 = Normalizer.normalize(levelValue.getValue(), Normalizer.Form.NFD).replaceAll("[\\p{InCombiningDiacriticalMarks}]", "");
                
                return value1.compareToIgnoreCase(value2);
            }
            else
            {
                return _order.compareTo(levelValue.getOrder());
            }
        }
    }
    
    private Cache<LevelValuesCacheKey, Map<String, LevelValue>> _getLevelValuesCache()
    {
        return _cacheManager.get(LEVEL_VALUES_CACHE);
    }
    
    private static final class LevelValuesCacheKey extends AbstractCacheKey
    {
        private LevelValuesCacheKey(String attributeName, String lang, String workspace)
        {
            super(attributeName, lang, workspace);
        }
        
        static LevelValuesCacheKey of(String attributeName, String lang, String workspace)
        {
            return new LevelValuesCacheKey(attributeName, lang, workspace);
        }
    }
}
