/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Default helper to export a {@link Program} to CDM-fr format.
 */
public class ExportCDMfrHelper implements Component, Serviceable, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = ExportCDMfrHelper.class.getName();
    
    /** The name of request parameter to use to get the valid version of content */
    public static final String REQUEST_PARAM_VALID_LABEL = "validLabel";
    
    /** The name of request parameter to set this export for a Ametys application */
    public static final String REQUEST_PARAM_EXPORT_FOR_AMETYS = "forAmetys";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The component for CDM export */
    protected ExportCDMfrManager _exportCDMfrManager;
    /** The Avalon context */
    protected Context _context;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _exportCDMfrManager = (ExportCDMfrManager) smanager.lookup(ExportCDMfrManager.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Generate the CDM-fr XML export for the current program.
     * @param contentHandler The content handler
     * @throws SAXException If an error occurs during sax
     */
    public void export(ContentHandler contentHandler) throws SAXException
    {
        Program program = _getProgram();
        
        _prepareRequest(program);
        
        _prepareResponse(program);
        
        _doExport(contentHandler, program);
    }
    
    /**
     * Do export
     * @param contentHandler The content handler to sax into
     * @param program the porogram to export
     * @throws SAXException If an error occurs during sax
     */
    protected void _doExport(ContentHandler contentHandler, Program program) throws SAXException
    {
        _exportCDMfrManager.generateCDM(contentHandler, program);
    }
    
    /**
     * Get the program to export
     * @return the program to export
     */
    protected Program _getProgram()
    {
        // Get the program from request
        Request request = _getRequest();

        String programId = request.getParameter("id");
        Program program = _resolver.resolveById(programId);
        
        // Add the program to the request
        request.setAttribute(Content.class.getName(), program);
        
        return program;
    }
    
    /**
     * Prepare the request before export
     * @param program the program to export
     */
    protected void _prepareRequest(Program program)
    {
        Request request = _getRequest();
        
        if (request.getParameter(REQUEST_PARAM_VALID_LABEL) != null)
        {
            request.setAttribute(ODFHelper.REQUEST_ATTRIBUTE_VALID_LABEL, true);
        }
        
        if (request.getParameter(REQUEST_PARAM_EXPORT_FOR_AMETYS) != null)
        {
            request.setAttribute(ExportCDMfrManager.REQUEST_ATTRIBUTE_EXPORT_FOR_AMETYS, true);
        }
    }
    
    /**
     * Prepare the response
     * @param program the program to export
     */
    protected void _prepareResponse(Program program)
    {
        Response response = ContextHelper.getResponse(_context);
        response.setHeader("Content-Disposition", "attachment; filename=\"" + program.getCDMId() + ".xml" + "\"");
    }
    
    /**
     * Get the request
     * @return the request
     */
    protected Request _getRequest()
    {
        return ContextHelper.getRequest(_context);
    }
}
