/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.course;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.content.CopyODFContentUpdater;
import org.ametys.odf.courselist.CourseList;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Copy updater to initialize shareable fields on {@link Course}.
 */
public class ShareableCourseCopyUpdater extends AbstractLogEnabled implements CopyODFContentUpdater, Serviceable
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The shareable course helper */
    protected ShareableCourseHelper _shareableCourseHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
    }
    
    @Override
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        for (Content copiedContent : copiedContents.values())
        {
            if (copiedContent instanceof Course copiedCourse)
            {
                // Get created parent course list of created course content
                CourseList courseListParent = copiedCourse.getParentCourseLists()
                        .stream()
                        .filter(c -> copiedContents.containsValue(c)) // filter on created course list by the copy
                        .findFirst()
                        .orElseGet(() -> (CourseList) targetParentContent);               
                
                if (_shareableCourseHelper.initializeShareableFields(copiedCourse, courseListParent, _currentUserProvider.getUser(), false))
                {
                    copiedCourse.saveChanges();
                    
                    // Create a new version
                    copiedCourse.checkpoint();
                }
            }
        }
    }
}
