/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.rights.ODFRightHelper.PermissionContext;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalWithPermissionContextAccessController;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link AccessController} for a ODF {@link Content}
 */
public class ODFContentHierarchicalAccessController extends AbstractHierarchicalWithPermissionContextAccessController<AmetysObject, ODFRightHelper.PermissionContext>
{
    /** the odf context category */
    public static final I18nizableText ODF_CONTEXT_CATEGORY = new I18nizableText("plugin.odf", "PLUGINS_ODF_RIGHT_ASSIGNMENT_CONTEXT_CONTENTS_LABEL");
    /** The helper for root content */
    protected ODFHelper _odfHelper;
    /** The helper for contents */
    protected ContentHelper _contentHelper;
    /** Ametys Object Resolver */
    protected AmetysObjectResolver _resolver;
    /** The right helper for ODF contents */
    protected ODFRightHelper _odfRightHelper;
    /** The ODF contents tree helper */
    protected ODFContentsTreeHelper _odfContentTreeHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfContentTreeHelper = (ODFContentsTreeHelper) manager.lookup(ODFContentsTreeHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _odfRightHelper = (ODFRightHelper) manager.lookup(ODFRightHelper.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof ProgramItem || object instanceof OrgUnit || object instanceof Person || object instanceof CoursePart || _isODFRootContent(object);
    }
    
    private boolean _isODFRootContent(Object object)
    {
        try
        {
            return object.equals(_odfHelper.getRootContent());
        }
        catch (UnknownAmetysObjectException e)
        {
            // The given object can not be the ODF root content, it does not exist
            return false;
        }
    }
    
    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object, PermissionContext permissionCtx)
    {
        if (!(object instanceof Content))
        {
            return null;
        }
        
        return _odfRightHelper.getParents((Content) object, permissionCtx);
    }
    
    @Override
    protected PermissionContext _getPermissionContext(AmetysObject initialObject)
    {
        return new PermissionContext((Content) initialObject);
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            return Collections.singleton(_odfHelper.getRootContent());
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        if (object instanceof Content)
        {
            Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
            if (object instanceof ProgramItem)
            {
                return new I18nizableText("plugin.odf", "PLUGINS_ODF_ODF_CONTENT_ACCESS_CONTROLLER_PROGRAM_ITEM_CONTEXT_LABEL", params);
            }
            else if (object instanceof CoursePart)
            {
                return new I18nizableText("plugin.odf", "PLUGINS_ODF_ODF_CONTENT_ACCESS_CONTROLLER_COURSE_PART_CONTEXT_LABEL", params);
            }
            else if (object instanceof OrgUnit)
            {
                return new I18nizableText("plugin.odf", "PLUGINS_ODF_ODF_CONTENT_ACCESS_CONTROLLER_ORGUNIT_CONTEXT_LABEL", params);
            }
            else if (object instanceof Person)
            {
                return new I18nizableText("plugin.odf", "PLUGINS_ODF_ODF_CONTENT_ACCESS_CONTROLLER_PERSON_CONTEXT_LABEL", params);
            }
        }
        else if (object.equals(_odfHelper.getRootContent()))
        {
            return new I18nizableText("plugin.odf", "PLUGINS_ODF_ODF_CONTENT_ACCESS_CONTROLLER_ROOT_CONTEXT_EXPLANATION_LABEL");
        }
        
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Content content)
        {
            return getContentObjectLabel(content, _odfContentTreeHelper);
        }
        else if (object.equals(_odfHelper.getRootContent()))
        {
            return new I18nizableText("plugin.odf", "PLUGINS_ODF_ODF_CONTENT_ACCESS_CONTROLLER_ROOT_CONTEXT_LABEL");
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        return ODF_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        if (object instanceof AmetysObjectCollection)
        {
            return 10;
        }
        return super.getObjectPriority(object);
    }

    /**
     * Get a standardized object label for an ODF content
     * @param content the content
     * @param helper  the ODF content tree helper
     * @return the label
     */
    public static I18nizableText getContentObjectLabel(Content content, ODFContentsTreeHelper helper)
    {
        String code = null;
        if (content instanceof ProgramItem programItem)
        {
            code = programItem.getDisplayCode();
        }
        else if (content instanceof OrgUnit orgunit)
        {
            code = orgunit.getUAICode();
        }
        
        if (StringUtils.isNotEmpty(code))
        {
            return new I18nizableText(content.getTitle() + " [" + code + "]");
        }
        else
        {
            return new I18nizableText(content.getTitle());
        }
    }
}
