/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.odf.ProgramItem;

import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract OSWorkflow function for creating a {@link ProgramItem} content
 * Populate catalog and code metadata.
 */
public abstract class AbstractCreateODFProgramItemFunction extends AbstractCreateODFContentFunction
{
    @Override
    protected void _populateAdditionalData(Map transientVars, ModifiableContent content) throws WorkflowException
    {
        super._populateAdditionalData(transientVars, content);
        
        if (content instanceof ProgramItem programItem)
        {
            /* Set the catalog */
            
            // Try to get the catalog from transient vars
            _getValueFromTransientVars(transientVars, CONTENT_CATALOG_KEY)
                // Use typed values from the EditContentFunction
                .or(() -> _getValueFromContext(transientVars, EditContentFunction.VALUES_KEY, ProgramItem.CATALOG))
                // Use raw values from the EditContentFunction
                .or(() -> _getValueFromContext(transientVars, EditContentFunction.FORM_RAW_VALUES, EditContentFunction.FORM_ELEMENTS_PREFIX + ProgramItem.CATALOG))
                // Used to get the catalog if the program item is created by java code, like for the csv import.
                .or(() -> _getValueFromInitialValueSupplier(transientVars, List.of(ProgramItem.CATALOG)))
                // Set the catalog if value is present
                .ifPresent(programItem::setCatalog);

            /* Set the code */
            
            // Used to get the code if the program item is created by java code, like for the csv import.
            _getValueFromInitialValueSupplier(transientVars, List.of(ProgramItem.CODE))
                // Set the code if value is present
                .ifPresent(programItem::setCode);
            
            // Generate a code if empty
            String code = programItem.getCode();
            if (StringUtils.isEmpty(code))
            {
                programItem.setCode(_odfHelper.generateUniqueCode(content.getTypes()[0]));
            }
        }
    }
    
    private Optional<String> _getValueFromTransientVars(Map transientVars, String attributeName)
    {
        return Optional.ofNullable(transientVars.get(attributeName))
            .map(String.class::cast)
            .filter(StringUtils::isNotEmpty);
    }
    
    private Optional<String> _getValueFromContext(Map transientVars, String parameter, String attributeName)
    {
        return Optional.ofNullable(getContextParameters(transientVars))
            .map(params -> params.get(parameter))
            .map(Map.class::cast)
            .map(values -> values.get(attributeName))
            .map(String.class::cast)
            .filter(StringUtils::isNotEmpty);
    }

    @SuppressWarnings("unchecked")
    private Optional<String> _getValueFromInitialValueSupplier(Map transientVars, List<String> attributePath)
    {
        return Optional.ofNullable(transientVars.get(CreateContentFunction.INITIAL_VALUE_SUPPLIER))
            .map(Function.class::cast)
            .map(function -> function.apply(attributePath))
            .map(String.class::cast)
            .filter(StringUtils::isNotEmpty);
    }
}
