/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.xslt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.enumeration.OdfReferenceTableHelper.SortField;

/**
 * DOM layer on reference table elements
 */
public class OdfReferenceTableElement extends AbstractAmetysElement
{
    private String _lang;
    private OdfReferenceTableHelper _refTableHelper;
    private String _refTableId;
    private boolean _ordered;
    private boolean _includeArchived;
    
    private List<OdfReferenceTableEntry> _children;
    
    
    /**
     * Constructor
     * @param refTableId The id of reference table
     * @param refTableHelper the reference table helper
     * @param lang the language to use for entry
     */
    public OdfReferenceTableElement(String refTableId, OdfReferenceTableHelper refTableHelper, String lang)
    {
        this(refTableId, refTableHelper, lang, false, true);
    }
    
    /**
     * Constructor
     * @param refTableId The id of reference table
     * @param refTableHelper the reference table helper
     * @param lang the language to use for entry
     * @param ordered true to get items sorted by their 'order' attribute
     * @param includeArchived true to include archived items
     */
    public OdfReferenceTableElement(String refTableId, OdfReferenceTableHelper refTableHelper, String lang, boolean ordered, boolean includeArchived)
    {
        _refTableHelper = refTableHelper;
        _refTableId = refTableId;
        _lang = lang;
        _ordered = ordered;
        _includeArchived = includeArchived;
    }
    
    @Override
    public String getTagName()
    {
        return "items";
    }
    
    List<OdfReferenceTableEntry> getChildren()
    {
        if (_children == null)
        {
            _children = _ordered ? _refTableHelper.getItems(_refTableId, _includeArchived, new SortField("order", true)) : _refTableHelper.getItems(_refTableId, _includeArchived);
        }
        
        return _children;
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> result = new HashMap<>();
        
        result.put("id", new AmetysAttribute("id", "id", null, _refTableId, this));
        
        return result;
    }
    
    @Override
    public Node getFirstChild()
    {
        if (_children == null)
        {
            _children = _ordered ? _refTableHelper.getItems(_refTableId, _includeArchived, new SortField("order", true)) : _refTableHelper.getItems(_refTableId, _includeArchived);
        }
        
        if (_children.isEmpty())
        {
            return null;
        }
        
        return new OdfReferenceTableEntryElement(_children.get(0), _lang, this);
    }
}
