/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.oai;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.runtime.config.Config;

/**
 * Base class for all OAI responses.
 */
public abstract class AbstractOAIResponseGenerator extends AbstractGenerator
{
    @Override
    public final void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();

        contentHandler.startPrefixMapping("", "http://www.openarchives.org/OAI/2.0/");
        contentHandler.startPrefixMapping("xsi", "http://www.w3.org/2001/XMLSchema-instance");
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation", "xsi:schemaLocation", "http://www.openarchives.org/OAI/2.0/ http://www.openarchives.org/OAI/2.0/OAI-PMH.xsd");
        XMLUtils.startElement(contentHandler, "OAI-PMH", atts);

        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        TimeZone tz = TimeZone.getTimeZone("UTC");
        formatter.setTimeZone(tz);
        XMLUtils.createElement(contentHandler, "responseDate", formatter.format(new Date()));
        
        doGenerate();

        XMLUtils.endElement(contentHandler, "OAI-PMH");

        contentHandler.endDocument();
    }
    
    /**
     * Actually generates the OAI response.
     * @throws IOException if an I/O error occurs
     * @throws SAXException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected abstract void doGenerate() throws IOException, SAXException, ProcessingException;

    /**
     * Generates SAX events representing an error.
     * @param errorCode the OAI error code.
     * @param errorLabel the error label.
     * @throws SAXException if an error occurs while SAXing the OAI error.
     */
    protected void generateError(String errorCode, String errorLabel) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("code", errorCode);
        XMLUtils.createElement(contentHandler, "error", atts, errorLabel);
    }
    
    /**
     * Returns the base URL of the OAI repository.
     * @return the base URL of the OAI repository.
     */
    protected String getURL()
    {
        String url = StringUtils.removeEndIgnoreCase(Config.getInstance().getValue("cms.url"), "index.html");
        
        StringBuilder result = new StringBuilder(url);
        
        if (!url.endsWith("/"))
        {
            result.append('/');
        }
        
        return result.append("_odf/OAI").toString();
    }
}
