/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.cdmfr.CDMFRHandler;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.Program;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract observer to handle CDMFR file when a odf content is validated
 */
public abstract class AbstractCDMFRObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The CDM-fr handler */
    protected CDMFRHandler _cdmfrHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _cdmfrHandler = (CDMFRHandler) manager.lookup(CDMFRHandler.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        if (isActive() && !_cdmfrHandler.isSuspended() && event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return content instanceof OrgUnit || content instanceof Person || content instanceof ProgramItem;
        }
        
        return false;
    }
    
    @Override
    public int getPriority()
    {
        return 3000;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        Set<Program> programs = getRelatedPrograms(content);
        doObserve(programs);
    }
    
    /**
     * Get the related programs from the content
     * @param content the content
     * @return the related program
     */
    protected Set<Program> getRelatedPrograms(Content content)
    {
        return _cdmfrHandler.getProgramParents(content)
                .stream()
                .filter(this::_hasLiveVersion)
                .collect(Collectors.toSet());
    }
    
    private boolean _hasLiveVersion(Program program)
    {
        // Get all labels
        String[] allLabels = program.getAllLabels();
        return Arrays.asList(allLabels).contains(CmsConstants.LIVE_LABEL);
    }
    
    /**
     * <code>true</code> if the observer is active
     * @return <code>true</code> if the observer is active
     */
    protected abstract boolean isActive();

    /**
     * Do observe for CDM-fr observer
     * @param programs the concerned programs
     */
    protected abstract void doObserve(Set<Program> programs);
    
}
