/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Odf plugin initialization class
 */
public class Init extends AbstractLogEnabled implements org.ametys.runtime.plugin.Init, Serviceable, Configurable
{
    /** The odf root node name */
    public static final String _ODF_ROOT_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":odf";

    /** The odf contents root node name */
    public static final String _ODF_CONTENTS_ROOT_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":contents";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The orgunit provider */
    protected RootOrgUnitProvider _orgUnitProvider;
    /** The workflow */
    protected WorkflowProvider _workflowProvider;
    /** The i18n utils */
    protected I18nUtils _i18Utils;

    private String _orgUnitWorkflowName;
    private String _orgUnitWorkflowId;
    private I18nizableText _orgUnitTitle;

    private ODFHelper _odfHelper;


    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _orgUnitProvider = (RootOrgUnitProvider) manager.lookup(RootOrgUnitProvider.ROLE);
        _i18Utils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _orgUnitWorkflowName = configuration.getChild("orgunit-workflow-name").getValue(null);
        _orgUnitWorkflowId = configuration.getChild("orgunit-workflow-init-action-id").getValue(null);
        _orgUnitTitle = I18nizableText.parseI18nizableText(configuration.getChild("orgunit-title"), "plugin.odf");
    }

    @Override
    public void init() throws Exception
    {
        // Create ODF root node if needed
        _odfHelper.getRootContent(true);
        
        OrgUnit rootOrgUnit = _orgUnitProvider.getRoot();
        
        if (rootOrgUnit != null)
        {
            // Set UAI code if changed
            String uaiCode = Config.getInstance().getValue("odf.root-orgunit.uaiCode");
            // The current code is different from the one in configuration
            if (!uaiCode.equals(rootOrgUnit.getUAICode()))
            {
                if (ExternalizableDataStatus.EXTERNAL.equals(rootOrgUnit.getStatus(OrgUnit.CODE_UAI)))
                {
                    // Change the status to local if needed...
                    rootOrgUnit.setStatus(OrgUnit.CODE_UAI, ExternalizableDataStatus.LOCAL);
                }
                
                // ...and set the value found in conf to the current (and local) value
                rootOrgUnit.setValue(OrgUnit.CODE_UAI, uaiCode);
                rootOrgUnit.saveChanges();
            }
        }
        else
        {
            // Create the root for orgunits
            Map<String, Object> inputs = new HashMap<>();
            inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, new LinkedHashMap<>());
            inputs.put(CreateContentFunction.CONTENT_NAME_KEY, OrgUnitFactory.ODF_ORGUNIT_ROOT_NODE);
            
            String title = _i18Utils.translate(_orgUnitTitle, Config.getInstance().getValue("odf.programs.lang"));
            inputs.put(CreateContentFunction.CONTENT_TITLE_KEY, title);
            inputs.put(CreateContentFunction.CONTENT_TYPES_KEY, new String[] {OrgUnitFactory.ORGUNIT_CONTENT_TYPE});
            inputs.put(CreateContentFunction.CONTENT_LANGUAGE_KEY, Config.getInstance().getValue("odf.programs.lang"));
            
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow();
            workflow.initialize(_orgUnitWorkflowName, Integer.parseInt(_orgUnitWorkflowId), inputs);
            
            @SuppressWarnings({"static-access", "unchecked"})
            Map<String, Object> workflowResult = (Map<String, Object>) inputs.get(AbstractContentWorkflowComponent.RESULT_MAP_KEY);
            String contentId = (String) workflowResult.get("contentId");
            
            rootOrgUnit = _resolver.resolveById(contentId);
            rootOrgUnit.setValue(OrgUnit.CODE_UAI, Config.getInstance().getValue("odf.root-orgunit.uaiCode"));
            rootOrgUnit.saveChanges();
        }
    }
}
