/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.ModelItem;

/**
 * Copy updater to update the micro skills on a macro skill and the macro skills on a {@link Program} and (@link Course}.
 */
public class SkillsCopyUpdater extends AbstractProgramItemAttributeCopyUpdater implements Serviceable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    public List<Content> getAdditionalContents(String catalogName)
    {
        List<Content> results = new ArrayList<>();
        results.addAll(_getContents(catalogName, SkillEditionFunction.MICRO_SKILL_TYPE));
        results.addAll(_getContents(catalogName, SkillEditionFunction.MACRO_SKILL_TYPE));
        return results;
    }
    
    public Map<Content, Content> copyAdditionalContents(String initialCatalogName, String newCatalogName)
    {
        // Only copy the skills if they are enabled
        if (!ODFSkillsHelper.isSkillsEnabled())
        {
            return Map.of();
        }
        
        Map<Content, Content> copiedContents = new HashMap<>();
        copiedContents.putAll(_copySkills(_getContents(initialCatalogName, SkillEditionFunction.MICRO_SKILL_TYPE), newCatalogName));
        copiedContents.putAll(_copySkills(_getContents(initialCatalogName, SkillEditionFunction.MACRO_SKILL_TYPE), newCatalogName));
        return copiedContents;
    }
    
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        // For every copied program, update its links from the original macro skills to the copied macro skills
        for (Content copiedContent : copiedContents.values())
        {
            try
            {
                if (copiedContent instanceof Program program)
                {
                    if (ODFSkillsHelper.isSkillsEnabled())
                    {
                        _updateContentAttribute(program, Program.SKILLS, copiedContents);
                        _updateContentAttribute(program, Program.TRANSVERSAL_SKILLS, copiedContents);
                        _updateContentAttribute(program, Program.BLOCKING_SKILLS, copiedContents);
                        program.saveChanges();
                    }
                }
                else if (copiedContent instanceof Course course)
                {
                    if (ODFSkillsHelper.isSkillsEnabled())
                    {
                        _updateContentAttribute(course, Course.ACQUIRED_MICRO_SKILLS + ModelItem.ITEM_PATH_SEPARATOR + Course.ACQUIRED_MICRO_SKILLS_SKILLS, copiedContents);
                        _updateContentAttribute(course, Course.ACQUIRED_MICRO_SKILLS + ModelItem.ITEM_PATH_SEPARATOR + Course.ACQUIRED_MICRO_SKILLS_PROGRAM, copiedContents);
                        course.saveChanges();
                    }
                }
                else if (copiedContent instanceof ModifiableContent modifiableCopiedContent)
                {
                    if (_contentTypesHelper.isInstanceOf(copiedContent, SkillEditionFunction.MACRO_SKILL_TYPE))
                    {
                        if (ODFSkillsHelper.isSkillsEnabled())
                        {
                            _updateContentAttribute(modifiableCopiedContent, "parentProgram", copiedContents);
                        }
                        _updateContentAttribute(modifiableCopiedContent, "microSkills", copiedContents);
                        modifiableCopiedContent.saveChanges();
                    }
                    else if (_contentTypesHelper.isInstanceOf(copiedContent, SkillEditionFunction.MICRO_SKILL_TYPE))
                    {
                        _updateContentAttribute(modifiableCopiedContent, "parentMacroSkill", copiedContents);
                        modifiableCopiedContent.saveChanges();
                    }
                }
            }
            catch (Exception e)
            {
                getLogger().error("An error occurred while updating the content '{}'", copiedContent.getId(), e);
            }
        }
    }
    
    private Map<Content, Content> _copySkills(List<DefaultContent> skills, String newCatalogName)
    {
        Map<Content, Content> copiedSkills = new HashMap<>();
        for (DefaultContent skill : skills)
        {
            // Copy the skill in the targeted catalog
            try
            {
                // Create the skill
                ModifiableContent newSkill = skill.copyTo((ModifiableTraversableAmetysObject) skill.getParent(), NameHelper.filterName(skill.getTitle()));
                
                // Set the new catalog
                newSkill.setValue("catalog", newCatalogName);
                newSkill.saveChanges();
                
                copiedSkills.put(skill, newSkill);
            }
            catch (AmetysRepositoryException e)
            {
                getLogger().error("Impossible to create the skill '{}' ({}) while creating the catalog {}", skill.getTitle(), skill.getId(), newCatalogName, e);
            }
        }
        
        return copiedSkills;
    }
    
    private <T extends Content> List<T> _getContents(String catalogName, String contentType)
    {
        return this.<T>_getContents(catalogName, contentType, null).toList();
    }
    
    private <T extends Content> Stream<T> _getContents(String catalogName, String contentType, Content skill)
    {
        AndExpression expression = new AndExpression();
        expression.add(new ContentTypeExpression(Operator.EQ, contentType));
        expression.add(new StringExpression("catalog", Operator.EQ, catalogName));
        if (skill != null)
        {
            expression.add(new StringExpression("code", Operator.EQ, skill.getValue("code")));
        }
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        return _resolver.<T>query(query).stream();
    }
}
