/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.rights.ContentAccessController;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessController.Permission.PermissionType;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Access controller for odf content that don't have an edition right, the edition right will be checked based on the creation right (example : skills).
 * 
 * <pre>
 * Configuration needed :
 * - 'content-type' : The content type
 *    -> The creation right will be retrieved from the content type definition
 * - 'i18n-keys-prefix' : The prefix of the i18n keys that will be used :
 *    -> <b>i18n-keys-prefix</b>_USER_ALLOWED_EXPLANATION
 *    -> <b>i18n-keys-prefix</b>_ANY_CONNECTED_ALLOWED_EXPLANATION
 *    -> <b>i18n-keys-prefix</b>_UNKNOWN_EXPLANATION
 *    -> <b>i18n-keys-prefix</b>_CONTENT_EXPLANATION_LABEL
 *    -> <b>i18n-keys-prefix</b>_ALL_CONTENTS_EXPLANATION_LABEL
 *    -> <b>i18n-keys-prefix</b>_ALL_CONTENTS_LABEL
 * </pre>
 */
public class EditContentWithCreationRightAccessController implements AccessController, Serviceable, Configurable, PluginAware
{
    /** Fake context to check on all contents */
    private static final String __ALL_CONTENTS_FAKE_CONTEXT = "$allContents$";
    
    private static List<String> _RIGHT_IDS = List.of("Workflow_Rights_Edition_Online");
    
    /** The contenttype to consider */
    protected String _contentType;
    /** The creation right found in the contenttype */
    protected String _creationRight;
    /** The prefix of the i18n keys to use to explain rights */
    protected String _i18nKeysPrefix;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    /** The right manager */
    protected RightManager _rightManager;

    private String _pluginName;
    private String _id;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _contentType = configuration.getChild("content-type").getValue();
        if (!_contentTypeExtensionPoint.hasExtension(_contentType))
        {
            throw new ConfigurationException("The content type '" + _contentType + "' is not registered in the content type extension point.");
        }
        else
        {
            ContentType contentType = _contentTypeExtensionPoint.getExtension(_contentType);
            _creationRight = contentType.getRight();
            
            if (_creationRight == null)
            {
                throw new ConfigurationException("The content type '" + _contentType + "' has not creation right in its definition.");
            }
        }
        
        
        _i18nKeysPrefix = configuration.getChild("i18n-keys-prefix").getValue();
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Content content && _contentTypesHelper.isInstanceOf(content, _contentType);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _id = id;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (_RIGHT_IDS.contains(rightId))
        {
            return _rightManager.hasRight(user, _creationRight, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (_rightManager.hasRight(user, _creationRight, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            _RIGHT_IDS.stream()
                .forEach(r -> {
                    permissionByRight.put(r, AccessResult.USER_ALLOWED);
                });
        }
        
        return permissionByRight;
    }
    
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (_RIGHT_IDS.contains(rightId))
        {
            for (UserIdentity user : _rightManager.getAllowedUsers(_creationRight, "/${WorkspaceName}").getAllowedUsers())
            {
                permissionByUser.put(user, AccessResult.USER_ALLOWED);
            }
        }
        return permissionByUser;
    }
    
    public AccessExplanation explainReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return _getAccessExplanation(getReadAccessPermissionForAnyConnectedUser(object), object);
    }
    
    public AccessExplanation explainReadAccessPermission(UserIdentity user, Set<GroupIdentity> groups, Object object)
    {
        return _getAccessExplanation(getReadAccessPermission(user, groups, object), object);
    }
    
    public AccessExplanation explainPermission(UserIdentity user, Set<GroupIdentity> groups, String rightId, Object object)
    {
        return _getAccessExplanation(getPermission(user, groups, rightId, object), object);
    }
    
    private AccessExplanation _getAccessExplanation(AccessResult permission, Object object)
    {
        switch (permission)
        {
            case USER_ALLOWED:
            case ANY_CONNECTED_ALLOWED:
            case UNKNOWN:
                return new AccessExplanation(getId(), permission,
                        new I18nizableText("plugin." + _pluginName, _i18nKeysPrefix + "_" + permission.name() + "_EXPLANATION",
                                Map.of(
                                        "objectLabel", getObjectLabelForExplanation(object)
                                    )
                                )
                        );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), permission);
        }
    }
    
    private I18nizableText getObjectLabelForExplanation(Object object)
    {
        if (object.equals(__ALL_CONTENTS_FAKE_CONTEXT))
        {
            return new I18nizableText("plugin." + _pluginName, _i18nKeysPrefix + "_ALL_CONTENTS_EXPLANATION_LABEL");
        }
        else if (object instanceof Content content)
        {
            return new I18nizableText(
                    "plugin." + _pluginName,
                    _i18nKeysPrefix + "_CONTENT_EXPLANATION_LABEL",
                    Map.of("title", new I18nizableText(content.getTitle()))
                );
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectLabel(Object object)
    {
        if (object.equals(__ALL_CONTENTS_FAKE_CONTEXT))
        {
            return new I18nizableText("plugin." + _pluginName, _i18nKeysPrefix + "_ALL_CONTENTS_LABEL");
        }
        return new I18nizableText(((Content) object).getTitle());
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        // Do not list all the contents.
        // Instead take advantage of the fact that the getPermission methods
        // ignore the context to call them with a fake context to generate a explanation for "all the content of the type"
        Map<Permission, AccessExplanation> contextPermissions = new HashMap<>();
        for (String rightId: _RIGHT_IDS)
        {
            AccessExplanation explanation = explainPermission(identity, groups, rightId, __ALL_CONTENTS_FAKE_CONTEXT);
            if (explanation.accessResult() != AccessResult.UNKNOWN)
            {
                contextPermissions.put(new Permission(PermissionType.RIGHT, rightId), explanation);
            }
        }
        
        return Map.of(getExplanationObject(__ALL_CONTENTS_FAKE_CONTEXT), contextPermissions);
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnyConnected(Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public Map<UserIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByUser(Object object)
    {
        Map<UserIdentity, Map<Permission, AccessExplanation>> permissionsByUser = new HashMap<>();
        
        // All user have the same set of permission. Compute it once and for all
        Map<Permission, AccessExplanation> userPermissions = new HashMap<>();
        AccessExplanation explanation = _getAccessExplanation(AccessResult.USER_ALLOWED, object);
        for (String right: _RIGHT_IDS)
        {
            Permission permission = new Permission(PermissionType.RIGHT, right);
            userPermissions.put(permission, explanation);
        }
        
        for (UserIdentity user : _rightManager.getAllowedUsers(_creationRight, "/${WorkspaceName}").getAllowedUsers())
        {
            permissionsByUser.put(user, userPermissions);
        }
        
        return permissionsByUser;
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return ContentAccessController.CONTENT_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        if (object.equals(__ALL_CONTENTS_FAKE_CONTEXT))
        {
            return 5;
        }
        return AccessController.super.getObjectPriority(object);
    }
    
    public String getId()
    {
        return _id;
    }
    
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Collections.EMPTY_MAP;
    }
    
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnonymous(Object object)
    {
        return Map.of();
    }
    
    public Map<GroupIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByGroup(Object object)
    {
        return Map.of();
    }
}
