/*
 *  Copyright 2026 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentFunction;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * If course part is modified, the course holder has to be in draft state too.
 */
public class DraftCourseHolderFunction extends AbstractContentFunction
{
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        // Retrieve current content
        WorkflowAwareContent content = getContent(transientVars);
        
        if (content instanceof CoursePart coursePart)
        {
            Course courseHolder = coursePart.getCourseHolder();
            
            /* If the course holder:
             *  - exists,
             *  - is not the initial caller of the current workflow action (by invert relations),
             *  - contains the current course part as a child,
             *  - is not in draft state,
             *  - has the action 22 available (move to draft)
             * Then move it to draft state.
             */
            if (courseHolder != null
                    && !courseHolder.getId().equals(getContextParameters(transientVars).get(EditContentFunction.INITIAL_CONTENT_ID))
                    && courseHolder.getCourseParts().contains(coursePart)
                    && courseHolder.getCurrentStepId() != 1
                    && _contentWorkflowHelper.isAvailableAction(courseHolder, 22))
            {
                _contentWorkflowHelper.editContent(courseHolder, Map.of(), 22);
            }
        }
    }
    
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_MOVE_DRAFT_COURSE_HOLDER_FUNCTION_LABEL");
    }
    
    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
}
