/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.generator;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;

/**
 * SAX user subscriptions
 */
public class UserTagSubscriptionsGenerator extends ServiceableGenerator
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user manager */
    protected UserManager _userManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "subscriptions");
        _saxBroadcastChannels();
        _saxFrequencies();
        XMLUtils.endElement(contentHandler, "subscriptions");
        
        contentHandler.endDocument();
    }
    
    /**
     * Sax broadcast channel
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxBroadcastChannels() throws SAXException 
    {
        XMLUtils.startElement(contentHandler, "broadcastChannels");
        for (BroadcastChannel broadcastChannel : BroadcastChannel.values())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", broadcastChannel.name());
            XMLUtils.startElement(contentHandler, "broadcastChannel", attrs);
            BroadcastChannelHelper.getLabel(broadcastChannel).toSAX(contentHandler, "label");
            BroadcastChannelHelper.getSmartLabel(broadcastChannel).toSAX(contentHandler, "smartLabel");
            XMLUtils.endElement(contentHandler, "broadcastChannel");
        }
        XMLUtils.endElement(contentHandler, "broadcastChannels");
    }
    
    /**
     * Sax frequencies
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxFrequencies() throws SAXException 
    {
        XMLUtils.startElement(contentHandler, "frequencies");
        for (Frequency frequency : Frequency.values())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", frequency.name());
            XMLUtils.startElement(contentHandler, "frequency", attrs);
            FrequencyHelper.getSmartLabel(frequency).toSAX(contentHandler, "smartLabel");
            XMLUtils.endElement(contentHandler, "frequency");
        }
        XMLUtils.endElement(contentHandler, "frequencies");
    }
}
