/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.mail;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;

/**
 * Component to build wrapped HTML mail body for report activities related to a tag subscription
 */
public class TagSubscriptionSummaryMailBodyHelper implements Contextualizable, Serviceable, Initializable
{
    /** The avalon context */
    protected static Context _context;
    
    private static SourceResolver _srcResolver;
    private static ContentTypesHelper _contentTypesHelper;

    private static TagSubscriptionSummaryMailBodyHelper __instance;

    @Override
    public void contextualize(Context context)
    {
        _context = context;
    }
    
    public void initialize() throws Exception
    {
        __instance = this;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }

    /**
     * Get the URI to resolve wrapped HTML email body
     * @return the uri
     */
    public String getMailBodyUri()
    {
        Request request = ContextHelper.getRequest(_context);
        
        String siteName = WebHelper.getSiteName(request);
        
        return "cocoon://_plugins/page-subscription/" + siteName + "/wrapped-mail/subscriptions/body.html";
    }
    
    /**
     * Creates a new mail body builder.
     * @return the newly created {@link MailBodyBuilder}.
     */
    public static MailBodyBuilder newHTMLBody()
    {
        return new MailBodyBuilder(__instance.getMailBodyUri());
    }
    
    /**
     * Implements the builder pattern for creating and sending emails.
     */
    public static class MailBodyBuilder
    {
        private Object _title;
        private List<Object> _messages;
        private Set<Content> _contents;
        private I18nizableText _linkText;
        private String _linkUrl;
        private I18nizableText _footerLinkText;
        private String _footerLinkUrl;
        private String _footerImgPluginName;
        private String _footerImgPath;
        private String _lang;
        private String _uri;
        private String _viewName = "abstract";
        private I18nizableText _hintMessage;
        
        MailBodyBuilder(String uri)
        {
            _uri = uri;
        }
        
        /**
         * Set the language
         * @param lang the language
         * @return this builder
         */
        public MailBodyBuilder withLanguage(String lang)
        {
            _lang = lang;
            return this;
        }
       
        /**
         * Set the title
         * @param title the title as String or {@link I18nizableText}
         * @return this builder
         */
        public MailBodyBuilder withTitle(Object title)
        {
            _title = title;
            return this;
        }
        
        /**
         * Set the hint message
         * @param hintMessage the hint message
         * @return this builder
         */
        public MailBodyBuilder withHint(I18nizableText hintMessage)
        {
            _hintMessage = hintMessage;
            return this;
        }
        
        
        /**
         * Set the view for contents
         * @param viewName the view name
         * @return this builder
         */
        public MailBodyBuilder withView(String viewName)
        {
            _viewName = viewName;
            return this;
        }
        
        /**
         * Set the main message
         * @param message the message as String or {@link I18nizableText}
         * @return this builder
         */
        public MailBodyBuilder withMessage(Object message)
        {
            _messages = List.of(message);
            return this;
        }
        
        /**
         * Add a new message
         * @param message the message as String or {@link I18nizableText}
         * @return this builder
         */
        public MailBodyBuilder addMessage(Object message)
        {
            if (_messages == null)
            {
                _messages = new ArrayList<>();
            }
            
            _messages.add(message);
            return this;
        }
        
        /**
         * Set the activities
         * @param contents the activities
         * @return this builder
         */
        public MailBodyBuilder withContents(Set<Content> contents)
        {
            _contents = contents;
            return this;
        }
        
        /**
         * Add a content
         * @param content the content
         * @return this builder
         */
        public MailBodyBuilder addContent(Content content)
        {
            if (_contents == null)
            {
                _contents = new HashSet<>();
            }
            
            _contents.add(content);
            return this;
        }
        
        /**
         * Add a main link to be displayed as a button
         * @param linkUrl the link url
         * @param linkText the link text
         * @return this builder
         */
        public MailBodyBuilder withLink(String linkUrl, I18nizableText linkText)
        {
            _linkText = linkText;
            _linkUrl = linkUrl;
            return this;
        }
        
        /**
         * Add a link to be displayed on footer
         * @param linkUrl the link url
         * @param linkText the link text
         * @return this builder
         */
        public MailBodyBuilder withFooterLink(String linkUrl, I18nizableText linkText)
        {
            return withFooterLink(linkUrl, linkText, null, null);
        }
        
        /**
         * Add a link to be displayed on footer
         * @param linkUrl the link url
         * @param linkText the link text
         * @param imgPluginName the plugin name that constains the image
         * @param imgPath the path of the image in the plugin
         * @return this builder
         */
        public MailBodyBuilder withFooterLink(String linkUrl, I18nizableText linkText, String imgPluginName, String imgPath)
        {
            _footerLinkText = linkText;
            _footerLinkUrl = linkUrl;
            _footerImgPluginName = imgPluginName;
            _footerImgPath = imgPath;
            return this;
        }
        
        /**
         * Build the body of the email as HTML
         * @return the HTML body
         * @throws IOException if an error occurred when building the email
         */
        public String build() throws IOException
        {
            Source src = null;
            Request request = ContextHelper.getRequest(_context);
            
            try
            {
                request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, toJson());
                
                Map<String, Object> parameters = new HashMap<>();
                parameters.put("lang", _lang);
                
                src = _srcResolver.resolveURI(_uri, null, parameters);
                Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
                return IOUtils.toString(reader);
            }
            finally
            {
                _srcResolver.release(src);
            }
        }
        
        /**
         * Get the JSON representation of the email ingredients
         * @return the JSON ingredients
         */
        public Map<String, Object> toJson()
        {
            Map<String, Object> json = new HashMap<>();
            
            json.put("title", _title);
            
            if (_hintMessage != null)
            {
                json.put("hint", _hintMessage);
            }
            
            if (_messages != null && !_messages.isEmpty())
            {
                json.put("message", _messages);
            }
            
            if (StringUtils.isNotEmpty(_linkUrl))
            {
                json.put("link", Map.of("url", _linkUrl, "text", _linkText));
            }
            
            if (_contents != null && !_contents.isEmpty())
            {
                json.put("content", _contents.stream()
                        .map(this::_content2json)
                        .toList());
            }
            
            json.put("footer", _footer2json());
            
            return json;
        }
        
        private Map<String, Object> _footer2json()
        {
            Map<String, Object> footer2json = new HashMap<>();
            
            if (_footerLinkUrl != null && _footerLinkText != null)
            {
                Map<String, Object> link2json = new HashMap<>();
                
                link2json.put("text", _footerLinkText);
                link2json.put("url", _footerLinkUrl);
                if (_footerImgPath != null && _footerImgPluginName != null)
                {
                    link2json.put("imgPluginName", _footerImgPluginName);
                    link2json.put("imgPath", _footerImgPath);
                }
                footer2json.put("link", link2json);
            }
            
            return footer2json;
        }
        
        private Map<String, Object> _content2json(Content content)
        {
            Map<String, Object> content2json = new HashMap<>();
            
            View view = _contentTypesHelper.getView(_viewName, content);
            if (view != null)
            {
                content2json.put("id", content.getId());
                
                // Values
                content2json.put("values", content.dataToJSON(view, DataContext.newInstance().withEmptyValues(false).withLocale(new Locale(_lang))));
            }
            
            if (content instanceof WebContent webContent)
            {
                String pageId = webContent.getReferencingPages()
                    .stream()
                    .findFirst()
                    .map(Page::getId)
                    .orElse(null);
                
                if (pageId != null)
                {
                    content2json.put("pageId", pageId);
                }
            }
            
            return content2json;
        }
        
    }
    
}
