/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.page;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.jcr.DefaultPage;

/**
 * DAO for page subscription
 *
 */
public class PageSubscriptionDAO extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = PageSubscriptionDAO.class.getName();
    
    private static final String __SUBSCRIBERS_PROPERTY_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":subscriberMails";

    private ObservationManager _observationManager;

    private CurrentUserProvider _currentUserProvider;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Get the subscribers on a page
     * @param page the page
     * @return the subscribers
     * @throws IllegalArgumentException if the page does not support subscription
     */
    public Set<String> getSubscribers(Page page) throws IllegalArgumentException
    {
        if (page instanceof JCRAmetysObject)
        {
            try
            {
                Node node = ((JCRAmetysObject) page).getNode();
                if (node.hasProperty(__SUBSCRIBERS_PROPERTY_NAME))
                {
                    Value[] values = node.getProperty(__SUBSCRIBERS_PROPERTY_NAME).getValues();
                    
                    try
                    {
                        return Arrays.stream(values)
                                .map(LambdaUtils.wrap(Value::getString))
                                .map(String::trim)
                                .filter(StringUtils::isNotEmpty)
                                .collect(Collectors.toSet());
                    }
                    catch (RuntimeException e)
                    {
                        // A wrapped RepositoryException occurs on #getString
                        getLogger().error("Unable to retrieve subscribers for page of id '" + page.getId() + "'", e.getCause());
                    }
                }
            }
            catch (RepositoryException e)
            {
                getLogger().error("Unable to retrieve subscribers for page of id '" + page.getId() + "'", e);
            }
        }
        else
        {
            throw new IllegalArgumentException("The page of id '" + page.getId() + "' does not support subscription");
        }
        
        return new HashSet<>();
    }
    
    /**
     * Determines if the given email is a subcriber of the given page
     * @param page The page
     * @param email The email to test
     * @return true if the given email is a subcriber of the given page
     */
    public boolean isSubscriber(Page page, String email)
    {
        return getSubscribers(page).contains(email);
    }
    
    /**
     * Add a subscriber to a page
     * @param page The page
     * @param email The email to add
     * @throws IllegalArgumentException if the page does not support subscription
     */
    public void addSubscriber(Page page, String email) throws IllegalArgumentException
    {
        Set<String> subscribers = getSubscribers(page);
        subscribers.add(email);
        setSubscribers(page, subscribers);
    }
    
    /**
     * Remove a subscriber from a page
     * @param page The page
     * @param email The email to remove
     * @throws IllegalArgumentException if the page does not support subscription
     */
    public void removeSubscriber(Page page, String email) throws IllegalArgumentException
    {
        Set<String> subscribers = getSubscribers(page);
        subscribers.remove(email);
        setSubscribers(page, subscribers);
    }
    
    /**
     * Set the subscribers to a page
     * @param page The page
     * @param subscribers The email of subscribers
     * @throws IllegalArgumentException if the page does not support subscription
     */
    public void setSubscribers(Page page, Set<String> subscribers) throws IllegalArgumentException
    {
        if (page instanceof DefaultPage)
        {
            try
            {
                Node node = ((DefaultPage) page).getNode();
                
                String[] subscribersAsStringArray = subscribers.toArray(new String[subscribers.size()]);
                node.setProperty(__SUBSCRIBERS_PROPERTY_NAME, subscribersAsStringArray);
                ((DefaultPage) page).saveChanges();
                
                // Notify observers that page data has been changed.
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(ObservationConstants.ARGS_PAGE, page);
                eventParams.put(ObservationConstants.ARGS_PAGE_ID, page.getId());
                _observationManager.notify(new Event(ObservationConstants.EVENT_PAGE_UPDATED, _currentUserProvider.getUser(), eventParams));
            }
            catch (RepositoryException e)
            {
                getLogger().error("Unable to set subscribers for page of id '" + page.getId() + "'", e);
            }
        }
        else
        {
            throw new IllegalArgumentException("The page of id '" + page.getId() + "' does not support subscription");
        }
        
    }
}
