/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.page.notifier;

import java.io.IOException;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.pagesubscription.page.PageSubscriptionDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.activities.AbstractPageActivityType;
import org.ametys.web.activities.notify.ActivityNotifier;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.jcr.DefaultPage;
import org.ametys.web.repository.site.Site;

/**
 * Abstract notifier for page activity
 */
public abstract class AbstractPageSubscriptionNotifier extends AbstractLogEnabled implements ActivityNotifier, Serviceable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;

    /** The user manager */
    protected UserManager _userManager;
    
    /** The subscription dao */
    protected PageSubscriptionDAO _subscriptionDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _subscriptionDAO = (PageSubscriptionDAO) manager.lookup(PageSubscriptionDAO.ROLE);
    }
    
    public List<String> getUsersEmailToNotify(Activity activity)
    {
        ActivityType activityType = activity.getActivityType();
        if (activityType instanceof AbstractPageActivityType)
        {
            User user = _userManager.getUser(activity.getAuthor());
            String activityAuthorEmail = user != null ? user.getEmail() : null;
            String pageId = activity.getValue(AbstractPageActivityType.PAGE_ID);
            Page page = _resolver.resolveById(pageId);
            
            if (page instanceof DefaultPage && isNotificationEnabled(page))
            {
                return _subscriptionDAO.getSubscribers(page)
                            .stream()
                            .filter(StringUtils::isNotBlank)
                            .filter(mail -> !mail.equals(activityAuthorEmail))
                            .toList();
            }
        }
        
        return List.of();
    }
    
    /**
     * Determines if the notification is currently enabled
     * @param page The page
     * @return true if the notification is enabled
     */
    protected boolean isNotificationEnabled(Page page)
    {
        Site site = page.getSite();
        return site.getValue(getSiteParameterId(), true, false);
    }
    
    /**
     * Id of the site parameter to enable/disable the notification
     * @return The id of site
     */
    protected abstract String getSiteParameterId();
    
    public String getMailSubject(Activity activity)
    {
        String pageId = activity.getValue(AbstractPageActivityType.PAGE_ID);
        Page page = _resolver.resolveById(pageId);
        
        I18nizableText i18nSubject = _getMailSubject(activity, page);
        
        return _i18nUtils.translate(i18nSubject, page.getSitemapName());
    }
    
    /**
     * Get the mail subject
     * @param activity the activity
     * @param page the page from activity
     * @return the mail subject
     */
    protected abstract I18nizableText _getMailSubject(Activity activity, Page page);

    public String getMailTextBody(Activity activity)
    {
        // No text body
        return null;
    }
    
    public String getMailHtmlBody(Activity activity)
    {
        String pageId = activity.getValue(AbstractPageActivityType.PAGE_ID);
        Page page = _resolver.resolveById(pageId);
      
        UserIdentity author = activity.getAuthor();
        User user = _userManager.getUser(author);
        
        if (user != null)
        {
            try
            {
                return _getMailHtmlBody(activity, page, user);
            }
            catch (Exception e)
            {
                getLogger().error("Can't get mail body for activity from event '{}' and type '{}'", activity.getEventType(), activity.getActivityType().getId(), e);
            }
        }
        return null;
    }
    
    /**
     * Get the mail body
     * @param activity the activity
     * @param page the page from activity
     * @param author the author of the activity
     * @return the mail body
     * @throws IOException if an error occurred
     */
    protected abstract String _getMailHtmlBody(Activity activity, Page page, User author) throws IOException;

    /**
     * Get the absolute URL of a page
     * @param page The page to retrieve the url
     * @return The absolute url of the page
     */
    protected String _getAbsolutePageUrl(Page page)
    {
        return new StringBuilder().append(page.getSite().getUrl())
                .append("/").append(page.getSitemapName())
                .append("/").append(page.getPathInSitemap())
                .append(".html").toString();
    }
    
}
