/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.schedulable.page;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.quartz.JobExecutionContext;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.notification.PageNotificationsHelper;
import org.ametys.plugins.pagesubscription.type.PageSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebConstants;
import org.ametys.web.activities.AbstractPageActivityType;
import org.ametys.web.mail.ReportActivitiesMailBodyHelper;
import org.ametys.web.mail.ReportActivitiesMailBodyHelper.MailBodyBuilder;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Schedulable to send not forced page notification summary by mail
 * The frequency is configurable. (DAILY, WEEKLY, MONTHLY)
 */
public class SendPageNotificationSummarySchedulable extends AbstractStaticSchedulable
{
    /** Tag for page that contains followed page configuration service */
    public static final String FOLLOWED_PAGES_CONFIG_PAGE_TAG = "TAG_FOLLOWED_PAGES_CONFIG";
    
    /** The frequency */
    protected Frequency _frequency;
    
    /** The mail subject */
    protected I18nizableText _mailSubject;
    
    /** The mail title */
    protected I18nizableText _mailTitle;
    
    /** The mail message */
    protected I18nizableText _mailMessage;
    
    /** The subscription type EP */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The page subscription type */
    protected PageSubscriptionType _pageSubscriptionType;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The page notification helper */
    protected PageNotificationsHelper _pageNotificationHelper;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The page DAO */
    protected PageDAO _pageDAO;
    
    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _pageSubscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _pageNotificationHelper = (PageNotificationsHelper) manager.lookup(PageNotificationsHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _frequency = Frequency.valueOf(configuration.getChild("frequency").getValue("DAILY"));
        _mailSubject = I18nizableText.parseI18nizableText(configuration.getChild("mail-subject"), "plugin." + _pluginName);
        _mailTitle = I18nizableText.parseI18nizableText(configuration.getChild("mail-title"), "plugin." + _pluginName);
        _mailMessage = I18nizableText.parseI18nizableText(configuration.getChild("mail-message"), "plugin." + _pluginName);
    }
    
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        // For each site
        for (Site site : _siteManager.getSites())
        {
            Request request = ContextHelper.getRequest(_context);
            
            // Retrieve the current context.
            RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
            try
            {
                // Force rendering context.FRONT to resolve URI
                _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
                request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, site.getName());
                
                Map<UserIdentity, Set<Activity>> pageActivitiesBySubscriber = _getPageActivitiesBySubscriber(site);
                for (UserIdentity subscriber : pageActivitiesBySubscriber.keySet())
                {
                    Set<Activity> pageActivities = pageActivitiesBySubscriber.get(subscriber);
                    if (!pageActivities.isEmpty())
                    {
                        String lang = _getLanguage(pageActivities);
                        
                        String subject = _i18nUtils.translate(_getI18nSubject(site), lang);
                        String body = _getMailBody(site, pageActivities, lang);
                        
                        User user = _userManager.getUser(subscriber);
                        if (user != null)
                        {
                            SendMailHelper.newMail()
                            .withRecipient(user.getEmail())
                            .withSubject(subject)
                            .withHTMLBody(body)
                            .withInlineCSS(false)
                            .withAsync(true)
                            .sendMail();
                        }
                    }
                }
            }
            finally 
            {
                // Restore current context
                _renderingContextHandler.setRenderingContext(currentContext);
            }
        }
    }
    
    private Map<UserIdentity, Set<Activity>> _getPageActivitiesBySubscriber(Site site)
    {
        Map<UserIdentity, Set<Activity>> pageActivitiesBySubscriber = new HashMap<>();
        for (Subscription subscription : _pageSubscriptionType.getSubscriptions(site, _frequency, BroadcastChannel.MAIL, null))
        {
            UserIdentity subscriber = subscription.getSubscriber().get(); // For page subscription, there is only user subscription (not group) ... so subscriber always exists
            Set<Activity> pages = pageActivitiesBySubscriber.getOrDefault(subscriber, new HashSet<>());
            
            String pageId = _pageSubscriptionType.getTarget(subscription);
            Activity lastActivity = _pageNotificationHelper.getLastActivity(site.getName(), pageId, subscription.getFrequency());
            if (lastActivity != null)
            {
                // Add page only if it has a recent activity for the current frequency
                pages.add(lastActivity);
            }
            pageActivitiesBySubscriber.put(subscriber, pages);
        }
        return pageActivitiesBySubscriber;
    }
    
    private String _getLanguage(Set<Activity> pageActivities)
    {
        // For the language, take the language of the first page
        return pageActivities.stream()
            .map(a -> a.<String>getValue(AbstractPageActivityType.PAGE_ID))
            .map(id -> (Page) _resolver.resolveById(id))
            .filter(Objects::nonNull)
            .map(Page::getSitemapName)
            .findFirst()
            .get();
    }
    
    /**
     * Get the subject of the mail
     * @param site 
     * @return the subject of the mail
     */
    protected I18nizableText _getI18nSubject(Site site)
    {
        Map<String, I18nizableTextParameter> params = Map.of(
                "site", new I18nizableText(site.getTitle())
            );
        return new I18nizableText(_mailSubject.getCatalogue(), _mailSubject.getKey(), params);
    }
    
    /**
     * Get the mail body of the mail
     * @param site the site
     * @param pages the set of pages with it last activity
     * @param lang the language
     * @return the mail body of the mail
     * @throws IOException if an error occurred
     */
    protected String _getMailBody(Site site, Set<Activity> pages, String lang) throws IOException
    {
        List<org.ametys.web.mail.ReportActivitiesMailBodyHelper.MailBodyBuilder.Activity> activities = pages.stream()
            .map(a -> this._wrapActivity(a, lang))
            .toList();
        
        MailBodyBuilder bodyBuilder =  ReportActivitiesMailBodyHelper.newHTMLBody()
            .withLanguage(lang)
            .withTitle(_mailTitle)
            .withMessage(_mailMessage)
            .withActivities(activities)
            .withLink(site.getUrl(), new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_NOTIFICATIONS_MAIL_SITE_LINK"));
        
        Optional<String> pageTagUrl = _pageDAO.findPagedIdsByTag(site.getName(), lang, FOLLOWED_PAGES_CONFIG_PAGE_TAG)
                .stream()
                .map(id -> ResolveURIComponent.resolve("page", id, false, true))
                .findFirst();
        if (pageTagUrl.isPresent())
        {
            bodyBuilder.withFooterLink(pageTagUrl.get(), new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_PAGE_NOTIFICATIONS_MAIL_PREFERENCE_LINK"), "core-ui", "img/mail/icon-letter.png");
        }
        
        return bodyBuilder.build();
    }
    
    private org.ametys.web.mail.ReportActivitiesMailBodyHelper.MailBodyBuilder.Activity _wrapActivity(Activity activity, String lang)
    {
        Map<String, I18nizableTextParameter> parameters = new HashMap<>();
        
        parameters.put("pageTitle", new I18nizableText(activity.getValue(AbstractPageActivityType.PAGE_TITLE)));
        
        String pageId = activity.getValue(AbstractPageActivityType.PAGE_ID);
        String pageUrl = ResolveURIComponent.resolve("page", pageId, false, true);
        parameters.put("pageUrl", new I18nizableText(pageUrl));
        
        I18nizableText messageI18n = new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_PAGE_NOTIFICATIONS_MAIL_CONTENT_MESSAGE", parameters);
        String message = _i18nUtils.translate(messageI18n, lang);
        
        // Activity is anonymized (no user avatar nor name)
        return new org.ametys.web.mail.ReportActivitiesMailBodyHelper.MailBodyBuilder.Activity(null, activity.getDate(), message);
    }
    
}
