/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.schedulable.tag;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.quartz.JobExecutionContext;

import org.ametys.cms.tag.CMSTag;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.web.repository.site.Site;

/**
 * Schedulable to send user tag notification summary by mail
 */
public class SendUserTagNotificationSummarySchedulable extends AbstractSendTagNotificationSummarySchedulable
{
    /** The frequency */
    protected Frequency _frequency;
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        _frequency = Frequency.valueOf(configuration.getChild("frequency").getValue("DAILY"));
    }
    
    @Override
    protected List<Site> getSiteToNotify(JobExecutionContext context)
    {
        return _siteManager.getSites().stream().toList();
    }
    
    @Override
    protected Frequency getFrequency(JobExecutionContext context)
    {
        return _frequency;
    }
    
    @Override
    protected FrequencyTiming getFrequencyTiming(JobExecutionContext context)
    {
        return new FrequencyTiming(
            FrequencyHelper.getDefaultFrequencyDay(), 
            FrequencyHelper.getDefaultFrequencyTime()
        );
    }
    
    @Override
    protected Map<CMSTag, Set<UserIdentity>> getSubscribersByTag(JobExecutionContext context, Site site, Frequency frequency)
    {
        Map<CMSTag, Set<UserIdentity>> subscribersByTag = new HashMap<>();
        
        // First get all subscribers for given site and frequency (resolve groups)
        Set<UserIdentity> subscribers = _tagSubscriptionType.getSubscribers(site, frequency, BroadcastChannel.MAIL, null);
        
        // FIXME Very low performance !! all users of a groups will retrieved and for each of them a JCR request will be done
        
        for (UserIdentity subscriber : subscribers)
        {
            // For each subscriber, get the computed subscriptions (taking into account user groups)
            // For example, if USER subscribed to THEME but he is part of GROUP subscribed (force) to THEME, #getUserSubscriptions will return only the group subscription
            List<Subscription> userSubscriptions = _tagSubscriptionType.getUserSubscriptions(site, frequency, BroadcastChannel.MAIL, subscriber, false, null)
                    .stream()
                    .filter(s -> !s.isForced()) // ignore forced group subscriptions (handled by another scheduler)
                    .toList();
            
            for (Subscription subscription : userSubscriptions)
            {
                CMSTag tag = _tagSubscriptionType.getTarget(subscription);
                Set<UserIdentity> users = subscribersByTag.getOrDefault(tag, new HashSet<>());
                users.add(subscriber);
                subscribersByTag.put(tag, users);
            }
        }
        
        return subscribersByTag;
    }
}
