/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.type;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.quartz.SchedulerException;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.group.GroupIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.TagSubscriptionContext;
import org.ametys.plugins.pagesubscription.schedulable.tag.ForcedSubscriptionSchedulerHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.web.repository.site.Site;

/**
 * Tag subscription type
 */
public class TagSubscriptionType extends AbstractSubscriptionType<TagSubscriptionContext, CMSTag>
{
    /** The id of the extension */
    public static final String ID = "subscription.tag";

    /** Data name for the tag of the subscription */
    public static final String TAG = "tag";

    /** The tag provider extension point */
    protected TagProviderExtensionPoint _tagProviderEP;
    
    /** The forced subscription scheduler helper */
    protected ForcedSubscriptionSchedulerHelper _forcedSubscriptionSchedulerHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _tagProviderEP = (TagProviderExtensionPoint) manager.lookup(TagProviderExtensionPoint.ROLE);
        _forcedSubscriptionSchedulerHelper = (ForcedSubscriptionSchedulerHelper) manager.lookup(ForcedSubscriptionSchedulerHelper.ROLE);
    }
    
    public CMSTag getTarget(Subscription subscription)
    {
        return _getTag(subscription);
    }
    
    @Override
    public Subscription forceSubscription(Site site, GroupIdentity group, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, TagSubscriptionContext context) throws Exception
    {
        Subscription subscription = super.forceSubscription(site, group, frequency, forcedTiming, broadcastChannels, context);
        _scheduleForcedSubscription(subscription);
        return subscription;
    }
    
    @Override
    public void editForceSubscription(Subscription subscription, Frequency frequency, List<BroadcastChannel> broadcastChannels, FrequencyTiming forcedTiming, TagSubscriptionContext context) throws Exception
    {
        _unscheduleForcedSubscription(subscription);
        super.editForceSubscription(subscription, frequency, broadcastChannels, forcedTiming, context);
        _scheduleForcedSubscription(subscription);
    }
    
    @Override
    public void unsubscribe(Subscription subscription) throws Exception
    {
        _unscheduleForcedSubscription(subscription);
        super.unsubscribe(subscription);
    }
    
    private void _unscheduleForcedSubscription(Subscription subscription) throws SchedulerException
    {
        if (_canCreateRunnable(subscription))
        {
            _forcedSubscriptionSchedulerHelper.unscheduleForcedSubscription(subscription);
        }
    }
    
    private void _scheduleForcedSubscription(Subscription subscription) throws SchedulerException
    {
        if (_canCreateRunnable(subscription))
        {
            _forcedSubscriptionSchedulerHelper.scheduleForcedSubscription(subscription);
        }
    }
    
    private boolean _canCreateRunnable(Subscription subscription)
    {
        // Create runnable for not instant frequency and mail channel
        return subscription.getFrequency() != Frequency.INSTANT
            && subscription.getBroadcastChannels()
                    .stream()
                    .filter(c -> c == BroadcastChannel.MAIL)
                    .findAny()
                    .isPresent();
    }
    
    @Override
    protected void setAdditionalData(Subscription subscription, TagSubscriptionContext context)
    {
        subscription.setValue(TAG, context.getTagName());
    }
    
    @Override
    protected boolean isSubscriptionValid(Subscription subscription)
    {
        // Check if tag still exist
        return _getTag(subscription) != null;
    }
    
    @Override
    protected Expression getAdditionalFilterExpression(TagSubscriptionContext context)
    {
        String tagName = context.getTagName();
        if (tagName != null)
        {
            return new StringExpression(TAG, Operator.EQ, tagName);
        }
        
        String rootTagName = context.getRootTagName();
        Map<String, Object> tagParams = Map.of("siteName", context.getSiteName());
        if (rootTagName != null && _tagProviderEP.hasTag(rootTagName, tagParams))
        {
            CMSTag tag = _tagProviderEP.getTag(rootTagName, tagParams);
            
            Expression[] tagsExpr = tag.getTags()
                .values()
                .stream()
                .map(t -> new StringExpression(TAG, Operator.EQ, t.getName()))
                .toArray(Expression[]::new);
            
            return new OrExpression(tagsExpr);
        }
        
        return null;
    }
    
    @Override
    public Map<String, Object> subscriptionToJSON(Subscription subscription)
    {
        Map<String, Object> subscriptionToJSON = super.subscriptionToJSON(subscription);
        
        CMSTag tag = _getTag(subscription);
        if (tag != null)
        {
            subscriptionToJSON.put("tag", Map.of(
                "label", tag.getTitle(),
                "name", tag.getName()
            ));
        }
        
        return subscriptionToJSON;
    }
    
    @Override
    protected void _saxAdditionalData(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        super._saxAdditionalData(contentHandler, subscription);
        
        CMSTag tag = _getTag(subscription);
        if (tag != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", tag.getName());
            XMLUtils.startElement(contentHandler, "tag", attrs);
            tag.getTitle().toSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "tag");
        }
    }
    
    private CMSTag _getTag(Subscription subscription)
    {
        String tagName = subscription.getValue(TAG);
        try
        {
            return _tagProviderEP.getTag(tagName, Map.of("siteName", subscription.getSite().getName()));
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred getting tag for name '{}'", tagName, e);
        }
        
        return null;
    }
    
    @Override
    protected String getUserPreferenceContextId(Subscription subscription)
    {
        return getTarget(subscription).getName();
    }
}
