/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
AmetysFront.UserPageSubscriptions = {
    
    SUBSCRIPTIONS_ATTR: "data-ametys-subscriptions",
    
    /** Attribute identifying a subscription wrapper */
    SUBSCRIPTION_ID_ATTR: "data-ametys-subscription-id",
    
    /** Attribute identifying a subscribed page  */
    SUBSCRIPTION_PAGE_ATTR: "data-ametys-subscription-page",
    
    /** The selector to get element receiving the number of subscribers */
    NB_SUBSCRIBERS_PAGE_SELECTOR: '[data-ametys-subscription-page-role="nb-subscribers"]',
    
    /** The selector to get element to subscribe */
    SUBSCRIBE_PAGE_SELECTOR: '[data-ametys-subscription-page-role="subscribe"]',
    
    /** The selector to get element to unsubscribe */
    UNSUBSCRIBE_PAGE_SELECTOR: '[data-ametys-subscription-page-role="unsubscribe"]',
    
    /** The selector to get element to display the number of subscribers */
    NB_SUBSCRIBERS_PAGE_SELECTOR: '[data-ametys-subscription-page-role="nb-subscribers"]',

    /** The selector to get element to display frequency */
    FREQUENCY_PAGE_SELECTOR: '[data-ametys-subscription-page-role="frequency"]',
    
    /** The selector to get element to display smart frequency */
    SMART_FREQUENCY_PAGE_SELECTOR: '[data-ametys-subscription-page-role="smart-frequency"]',

    /** The selector to get element that contains the list of subscribed pages */
    SUBSCRIPTION_PAGE_RESULTS_SELECTOR: '[data-ametys-subscription-page-role="results"]',
    
    /** The selector to get element to display when there is no subscribed page */
    SUBSCRIPTION_PAGE_EMPTY_SELECTOR: '[data-ametys-subscription-page-role="no-result"]',
    
    SUBSCRIPTION_PAGE_STATUS_SELECTOR: '[data-ametys-subscription-page-role="status"]',
    
    /** Attribute identifying a subscribed page for notification */
    PAGE_NOTIFICATION_ID_ATTR:"data-ametys-page-notification-id",
    
    /** Selector to get element with the date of a notification */
    PAGE_NOTIFICATION_DATE_SELECTOR: '[data-ametys-page-notification-role="date"]',
    
    /** Selector to get element with the author's name of a notification */
    PAGE_NOTIFICATION_AUTHOR_NAME_SELECTOR: '[data-ametys-page-notification-role="author-name"]',
    
    /** Selector to get the IMG element with the author's image of a notification */
    PAGE_NOTIFICATION_AUTHOR_IMG_SELECTOR: '[data-ametys-page-notification-role="author-img"]',
    
    
    /**
     * Get subscriptions of a page
     * @param {String} pageId the page id
     * @param {Function} callback the callback
     * @private
     */
    getSubscriptions: function(pageId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.PageSubscriptionsDAO',
            methodName: 'getPageSubscriptions',
            parameters: [pageId],
            callback: {
                handler: this._getSubscriptionsCb,
                scope: this,
                arguments: {
                    pageId : pageId,
                    callback: callback
                }
            },
            errorMessage: false
        });
    },
    
    /**
     * Callback function invoked after getting subscriber info
     * @param {Object} response the server response
     * @param {Object[]} args the additional arguments
     * @private
     */
    _getSubscriptionsCb: function(response, args)
    {
        this._updateHTML(args.pageId, response);
        
        if (typeof(args.callback) == 'function')
        {
            args.callback(args.pageId, response);
        }
    },
    
    /**
     * Susbcribe to a page
     * @param {String} pageId the page id
     * @param {Function} [callback] the callback function to call after subscribe
     * @private
     */
    subscribePage: function(pageId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.PageSubscriptionsDAO',
            methodName: 'subscribePage',
            parameters: [pageId],
            callback: {
                handler: this._subscribePageCb,
                scope: this,
                arguments: {
                    pageId: pageId,
                    callback: callback
                }
            },
            errorMessage : true
        });
    },
    
    /**
     * Callback function invoked after subscribing
     * @param {Object} response the server response
     * @param {Object[]} args the additional arguments
     * @private
     */
    _subscribePageCb: function(response, args)
    {
        if (response.success)
        {
            this._updateHTML(args.pageId, response);
            if (typeof(args.callback) == 'function')
	        {
	            args.callback(args.pageId, response);
	        }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_SUBSCRIBE_ERROR}}", "{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_ERROR_TITLE}}");
        }
    },
    
    /**
     * Unsusbcribe a the page
     * @param {String} pageId the page id
     * @param {Function} [callback] the callback function to call after unsubscribe
     * @private
     */
    unsubscribePage: function(pageId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.PageSubscriptionsDAO',
            methodName: 'unsubscribePage',
            parameters: [pageId],
            callback: {
                handler: this._unsubscribePageCb,
                scope: this,
                arguments: {
                    pageId: pageId,
                    callback: callback
                }
            }
        });
    },
    
    /**
     * Callback function invoked after subscribing
     * @param {Object} response the server response
     * @param {Object[]} args the additional arguments
     * @private
     */
    _unsubscribePageCb: function(response, args)
    {
        if (response.success)
        {
            this._updateHTML(args.pageId, response);
            if (typeof(args.callback) == 'function')
	        {
	            args.callback(args.pageId, response);
	        }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_UNSUBSCRIBE_ERROR}}", "{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_ERROR_TITLE}}")
        }
    },
    
    /**
     * Update HTML view
     * @param {String} pageId the server response
     * @param {Object} data the page data
     * @private
     */
    _updateHTML: function(pageId, data)
    {
        let $wrapper = $j('[' + AmetysFront.UserPageSubscriptions.SUBSCRIPTION_PAGE_ATTR + '="' + pageId + '"]');
        
        if (data.hasSubscribed)
        {
            $wrapper.find(AmetysFront.UserPageSubscriptions.SUBSCRIBE_PAGE_SELECTOR).hide();
            $wrapper.find(AmetysFront.UserPageSubscriptions.UNSUBSCRIBE_PAGE_SELECTOR).show();
        }
        else
        {
            $wrapper.find(AmetysFront.UserPageSubscriptions.SUBSCRIBE_PAGE_SELECTOR).show();
            $wrapper.find(AmetysFront.UserPageSubscriptions.UNSUBSCRIBE_PAGE_SELECTOR).hide();
        }
        
        $wrapper.find(AmetysFront.UserPageSubscriptions.NB_SUBSCRIBERS_PAGE_SELECTOR).html(data.nbSubscribers);
    },
    
    /**
     * Edit subscription
     * @param {String} uniqueId the unique id to target wrapper element
     * @param {String} subscriptionId the subscription id
     * @param {String} frequency the frequency
     * @param {String[]} broadcastChannels the broadcast channels
     * @param {Function} [callback] the callback function to invoked after editing subscription
     * @private
     */
    editSubscription: function(uniqueId, subscriptionId, frequency, broadcastChannels, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.PageSubscriptionsDAO',
            methodName: 'editSubscription',
            parameters: [subscriptionId, frequency, broadcastChannels],
            callback: {
                handler: this._editSubscriptionCb,
                scope: this,
                arguments: {
                    uniqueId: uniqueId,
                    subscriptionId: subscriptionId,
                    callback: callback
                }
            }
        });
    },
    
    /**
     * Edit subscription callback
     * @param {Object} response the response
     * @param {Object[]} args the additional arguments
     * @private
     */
    _editSubscriptionCb: function(response, args)
    {
        if (response.success)
        {
            this._updateSubcription(args.uniqueId, response.subscription);
        
            if (typeof(args.callback) == 'function')
            {
                args.callback(args.subscriptionId, response);
            }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_EDIT_SUBSCRIPTION_ERROR}}", "{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_ERROR_TITLE}}");
        }
    },
    
    /**
     * @param {String} uniqueId the unique id to target wrapper element
     * @param {Object} subscription the subscription
     * @private
     */
    _updateSubcription: function(uniqueId, subscription)
    {
        let $subscriptionEl = $j('[' + AmetysFront.UserPageSubscriptions.SUBSCRIPTION_ID_ATTR + '="' + subscription.id + '"]');
        $subscriptionEl.find(AmetysFront.UserPageSubscriptions.FREQUENCY_PAGE_SELECTOR).html(subscription.frequency.label);
        $subscriptionEl.find(AmetysFront.UserPageSubscriptions.SMART_FREQUENCY_PAGE_SELECTOR).html(subscription.frequency.smartLabel);
        
        let $wrapper = $j('[' + AmetysFront.UserPageSubscriptions.SUBSCRIPTIONS_ATTR + '="' + uniqueId + '"]');
        $wrapper.find(AmetysFront.UserPageSubscriptions.SUBSCRIPTION_PAGE_STATUS_SELECTOR).html("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_EDIT_SUBSCRIPTION_SUCCESS}}");
    },
    
    /**
     * Unsubscribe
     * @param {String} uniqueId the unique id to target wrapper element
     * @param {String} subscriptionId the subscription id
     * @param {Function} [callback] the callback function to invoked after removing subscription
     * @param {Boolean} [skipConfirm] true to skip confirm before unsubscribe
     */
    unsubscribe: function (uniqueId, subscriptionId, callback, skipConfirm)
    {
        let me = this;
        
        function __internalUnsubscribe()
        {
            AmetysFront.ServerComm.callMethod({
                role: 'org.ametys.plugins.pagesubscription.dao.PageSubscriptionsDAO',
                methodName: 'unsubscribe',
                parameters: [subscriptionId],
                callback: {
                    handler: me._unsubscribeCb,
                    scope: me,
                    arguments: {
                        uniqueId: uniqueId,
                        subscriptionId: subscriptionId,
                        callback: callback
                    }
                },
                errorMessage: true
            });
        }
        
        if (skipConfirm)
        {
            __internalUnsubscribe()
        }
        else
        {
            AmetysFront.Utils.confirm("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_UNSUBSCRIBE_CONFIRM_TITLE}}", "{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_UNSUBSCRIBE_CONFIRM_MSG}}", function () {
                __internalUnsubscribe()
            });
        }
    },
    
    /**
     * Callback function invoked after unsubscribing
     * @param {Boolean} success the server response success
     * @param {Object[]} args the additional arguments
     * @private
     */
    _unsubscribeCb: function(success, args)
    {
        if (success)
        {
            let $subcriptionEl = $j('[' + AmetysFront.UserPageSubscriptions.SUBSCRIPTION_ID_ATTR + '="' + args.subscriptionId + '"]');
            $subcriptionEl.remove();
            
            let $wrapper = $j('[' + AmetysFront.UserPageSubscriptions.SUBSCRIPTIONS_ATTR + '="' + args.uniqueId + '"]');
            let $result = $wrapper.find(AmetysFront.UserPageSubscriptions.SUBSCRIPTION_PAGE_RESULTS_SELECTOR);
            let count = $result.children().length;
            
            $result.attr("tabindex", -1);
            $result.focus();
            
            if (count == 0)
            {
                $wrapper.find(AmetysFront.UserPageSubscriptions.SUBSCRIPTION_PAGE_EMPTY_SELECTOR).show();
            }
            $wrapper.find(AmetysFront.UserPageSubscriptions.SUBSCRIPTION_PAGE_STATUS_SELECTOR).html("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_UNSUBSCRIBE_SUCCESS}}")
            
            if (typeof(args.callback) == 'function')
	        {
	            args.callback(args.subscriptionId);
	        }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_UNSUBSCRIBE_ERROR}}", "{{i18n plugin.page-subscription:PAGE_SUBSCRIPTION_HELPER_ERROR_TITLE}}")
        }
    },
    
    /**
     * Get the user unread pages
     * @param {Function} callback the callback function to invoked after getting unread pages
     */
    getUnreadPages: function(selector, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.notification.PageNotificationsHelper',
            methodName: 'getUnreadPages',
            parameters: [AmetysFront.getAppParameters().siteName],
            callback: {
                handler: this._getUnreadPages,
                scope: this,
                arguments: {
                    selector: selector,
                    callback: callback
                }
            }
        });
    },
    
    /**
     * @private
     * Callback function after getting unread pages
     * @param {Object[]} unreadPages the unread pages
     * @param {Object} args the callback arguments
     */
    _getUnreadPages: function (unreadPages, args)
    {
        var $wrap = $j(args.selector);
        $wrap.find('[' + AmetysFront.UserPageSubscriptions.PAGE_NOTIFICATION_ID_ATTR + ']').hide();

        $j.each(unreadPages, function(i, unreadPage) {
            let $page = $wrap.find('[' + AmetysFront.UserPageSubscriptions.PAGE_NOTIFICATION_ID_ATTR + '="' + unreadPage.pageId + '"]');
            $page.find(AmetysFront.UserPageSubscriptions.PAGE_NOTIFICATION_DATE_SELECTOR).html(AmetysFront.Utils.toMomentDate(unreadPage.lastActivityDate));
            $page.find(AmetysFront.UserPageSubscriptions.PAGE_NOTIFICATION_AUTHOR_NAME_SELECTOR).html(unreadPage.author.fullname);
            $page.find(AmetysFront.UserPageSubscriptions.PAGE_NOTIFICATION_AUTHOR_IMG_SELECTOR).attr("src", unreadPage.author.imgUrl);
            $page.show();
        });
        
        if (typeof(args.callback) == 'function')
        {
            args.callback(unreadPages)
        }
    },
    
    /**
     * Mark a page as read
     * @param {String} pageId the page id
     * @param {Function} [callback] a callback function to invoked after marking as read
     */
    markPageAsRead: function(pageId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.notification.PageNotificationsHelper',
            methodName: 'markPageAsRead',
            parameters: [pageId],
            callback: {
                handler: function() {
                    if (typeof(callback) == 'function')
                    {
                        callback()
                    }
                }
            }
        });
    }
}
