/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import javax.jcr.Node;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.type.SubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * Object representing an subscription
 */
public class Subscription extends SimpleAmetysObject<SubscriptionFactory> implements ModifiableModelLessDataAwareAmetysObject
{
    /** key used to store the activity id when serialized */
    public static final String ACTIVITY_ID_KEY = "id";
    /** key used to store the activity name when serialized */
    public static final String ACTIVITY_NAME_KEY = "name";

    /**
     * Creates an {@link Subscription}.
     * @param node the node backing this {@link Subscription}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the factory which created the activity
     */
    public Subscription(Node node, String parentPath, SubscriptionFactory factory)
    {
        super(node, parentPath, factory);
    }

    public ModifiableModelLessDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getElementTypesExtensionPoint(), repositoryData);
    }
    
    /**
     * Get the type of the subscription
     * @return the subscription type
     */
    public SubscriptionType getSubscriptionType()
    {
        return _getFactory().getSubscriptionType(getValue(SubscriptionFactory.SUBSCRIPTION_TYPE_ID));
    }
    
    /**
     * Get the subscriber of the subscription
     * @return the subscriber subscription
     */
    public Optional<UserIdentity> getSubscriber()
    {
        return Optional.ofNullable(getValue(SubscriptionFactory.SUBSCRIBER));
    }
    
    /**
     * Get the subscribers group of the subscription
     * @return the subscribers group subscription
     */
    public Optional<GroupIdentity> getSubscribersGroup()
    {
        String value = getValue(SubscriptionFactory.SUBSCRIBERS_GROUP);
        return StringUtils.isNoneBlank(value)
                ? Optional.of(GroupIdentity.stringToGroupIdentity(value))
                : Optional.empty();
    }
    
    /**
     * Get the list of broadcast channel of the subscription
     * @return the list of broadcast channel
     */
    public List<BroadcastChannel> getBroadcastChannels()
    {
        return Stream.of(getValue(SubscriptionFactory.BROADCAST_CHANNEL, new String[0]))
            .map(c -> BroadcastChannel.valueOf(c))
            .toList();
    }
    
    /**
     * Get the frequency of the subscription
     * @return the frequency
     */
    public Frequency getFrequency()
    {
        String frequencyType = getValue(SubscriptionFactory.FREQUENCY);
        return Frequency.valueOf(frequencyType);
    }
    
    /**
     * <code>true</code> if the subscription is forced
     * @return <code>true</code> if the subscription is forced
     */
    public boolean isForced()
    {
        return getValue(SubscriptionFactory.IS_FORCED, false);
    }
    
    /**
     * Get the frequency timing hour if the subscription is forced. 
     * @return the frequency timing
     */
    public FrequencyTiming getForceFrequencyTiming()
    {
        return new FrequencyTiming(
            getValue(SubscriptionFactory.FORCED_DAY),
            getValue(SubscriptionFactory.FORCED_HOUR)
        );
    }
    
    /**
     * Get the date of the subscription.
     * @return the date of the subscription.
     */
    public ZonedDateTime getDate()
    {
        return getValue(SubscriptionFactory.DATE);
    }
    
    /**
     * Get the site of the subscription
     * @return the site
     */
    public Site getSite()
    {
        AmetysObject parent = getParent();
        while (!(parent instanceof Site))
        {
            parent = parent.getParent();
        }

        return (Site) parent;
    }
}
