/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.dao;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.SubscriptionException;
import org.ametys.plugins.pagesubscription.context.TagSubscriptionContext;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.config.Config;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * DAO for force tag subscriptions
 */
public class ForceTagSubscriptionsDAO extends AbstractSubscriptionsDAO implements Initializable
{
    private static final String _FORCE_TAG_SUBSCRIPTION_RIGHT_ID = "Page_Subscription_Rights_Force_Subscriptions"; 
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The tag subscription type */
    protected TagSubscriptionType _tagSubscriptionType;
    
    private Long _day;
    private String _hour;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _day = Config.getInstance().getValue("plugin.web.subscription.day", true, 2L);
        _hour = Config.getInstance().getValue("plugin.web.subscription.hour", true, "10:00");
    }
    
    /**
     * Get the frequency information
     * @return the frequency information as json
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getFrequencyTiming()
    {
        return Map.of(
            "day", _day,
            "hour", _hour
        );
    }
    
    /**
     * Force the group subscription
     * @param siteName the site name
     * @param groupId the group id
     * @param tagName the tag name
     * @param frequency the frequency
     * @param forcedDay the day
     * @param forcedHour the hour
     * @param broadcastChannelNames the broadcast channels
     * @return the subscription as json
     * @throws Exception if an error occurred
     */
    @Callable(rights = _FORCE_TAG_SUBSCRIPTION_RIGHT_ID, context =  "/${WorkspaceName}")
    public Map<String, Object> forceGroupSubscription(String siteName, String groupId, String tagName, String frequency, String forcedDay, String forcedHour, List<String> broadcastChannelNames) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            Site site = _siteManager.getSite(siteName);
    
            GroupIdentity group = GroupIdentity.stringToGroupIdentity(groupId);
            TagSubscriptionContext context = TagSubscriptionContext.newInstance()
                    .withTag(tagName);
    
            List<BroadcastChannel> broadcastChannels = broadcastChannelNames.stream()
                    .map(c -> BroadcastChannel.valueOf(c))
                    .toList();
            
            Long day = StringUtils.isNotBlank(forcedDay) ? Long.valueOf(forcedDay) : _day;
            String hour = StringUtils.isNotBlank(forcedHour) ? forcedHour : _hour;
            
            Subscription subscription = _tagSubscriptionType.forceSubscription(site, group, Frequency.valueOf(frequency), new FrequencyTiming(day, hour), broadcastChannels, context);
            
            return Map.of("success", true, "subscription", _tagSubscriptionType.subscriptionToJSON(subscription));
        }
        catch (SubscriptionException e) 
        {
            getLogger().error("Unable to force group subscription for site name '{}' and group id '{}'", siteName, groupId, e);
            return Map.of("success", false, "type", e.getType().name());
        }
        catch (Exception e)
        {
            getLogger().error("Unable to force group subscription for site name '{}' and group id '{}'", siteName, groupId, e);
            return Map.of("success", false);
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Get all force group subscriptions 
     * @param siteName the site name
     * @return the list of subscriptions as json
     */
    @Callable(rights = _FORCE_TAG_SUBSCRIPTION_RIGHT_ID, context =  "/${WorkspaceName}")
    public List<Map<String, Object>> getForceGroupSubscriptions(String siteName)
    {
        Site site = _siteManager.getSite(siteName);
        
        return _tagSubscriptionType.getGroupsSubscriptions(site, Set.of(), true, TagSubscriptionContext.newInstance())
            .stream()
            .map(_tagSubscriptionType::subscriptionToJSON)
            .toList();
    }
    
    @Override
    @Callable(rights = _FORCE_TAG_SUBSCRIPTION_RIGHT_ID, context =  "/${WorkspaceName}")
    public Map<String, Object> getSubscription(String subscriptionId)
    {
        return getSubscription(subscriptionId, false);
    }
    
    /**
     * Edit the force subscription with the given id
     * @param subscriptionId the subscription id
     * @param frequency the frequency 
     * @param forcedDay the forced day
     * @param forcedHour the forced hour
     * @param broadcastChannels the broadcast channels
     * @return the subscription as JSON
     */ 
    @Callable(rights = _FORCE_TAG_SUBSCRIPTION_RIGHT_ID, context =  "/${WorkspaceName}")
    public Map<String, Object> editForceSubscription(String subscriptionId, String frequency, String forcedDay, String forcedHour, List<String> broadcastChannels)
    {
        Long day = StringUtils.isNotBlank(forcedDay) ? Long.valueOf(forcedDay) : _day;
        String hour = StringUtils.isNotBlank(forcedHour) ? forcedHour : _hour;
        
        return editSubscription(subscriptionId, frequency, new FrequencyTiming(day, hour), broadcastChannels, false);
    }
    
    @Override
    @Callable(rights = _FORCE_TAG_SUBSCRIPTION_RIGHT_ID, context =  "/${WorkspaceName}")
    public boolean unsubscribe(String subscriptionId) throws Exception
    {
        return unsubscribe(subscriptionId, false);
    }
}
