/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.observation;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.PageSubscriptionContext;
import org.ametys.plugins.pagesubscription.type.PageSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;

/**
 * Observer to unsubscribe to a page when the page is deleted
 */
public class RemoveSubscriptionOnPageDeletedObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_PAGE_DELETING);
    }

    @Override
    public int getPriority(Event event)
    {
        return MAX_PRIORITY;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            String pageId = (String) event.getArguments().get(ObservationConstants.ARGS_PAGE_ID);
            
            PageSubscriptionType pageSubscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);
            Page page = _resolver.resolveById(pageId);
            
            PageSubscriptionContext context = PageSubscriptionContext.newInstance().withPage(page);
            List<Subscription> subscriptions = pageSubscriptionType.getSubscriptions(page.getSite(), null, null, context);
            for (Subscription subscription : subscriptions)
            {
                // unsubscribe
                pageSubscriptionType.unsubscribe(subscription);
            }
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
}
