/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.observation;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.TagSubscriptionContext;
import org.ametys.plugins.pagesubscription.mail.TagSubscriptionSummaryMailBodyHelper;
import org.ametys.plugins.pagesubscription.mail.TagSubscriptionSummaryMailBodyHelper.MailBodyBuilder;
import org.ametys.plugins.pagesubscription.schedulable.tag.AbstractSendTagNotificationSummarySchedulable;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ObservationConstants;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.activities.AbstractPageActivityType;
import org.ametys.web.activities.PageUpdatedActivityType;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.site.Site;

import jakarta.mail.MessagingException;

/**
 * Observer to send instant tag notification on created activity of type {@link PageUpdatedActivityType}
 */
public class SendTagNotificationOnCreatedActivityObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable, Contextualizable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;

    /** The user manager */
    protected UserManager _userManager;
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The tag susbcription type */
    protected TagSubscriptionType _tagSubscriptionType;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The page DAO */
    protected PageDAO _pageDAO;

    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    
    /** The context */
    protected Context _context;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public boolean supports(Event event)
    {
        if (ObservationConstants.EVENT_ACTIVITY_CREATED.equals(event.getId()))
        {
            Activity activity = (Activity) event.getArguments().get(ObservationConstants.ARGS_ACTIVITY);
            return activity != null && activity.getActivityType() instanceof PageUpdatedActivityType;
        }
        
        return false;
    }

    @Override
    public int getPriority(Event event)
    {
        return MIN_PRIORITY;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        // Get activity by id because we lose JCR sessions with Activity object in the arguments
        String activityId = (String) event.getArguments().get(ObservationConstants.ARGS_ACTIVITY_ID);
        Activity activity = _resolver.resolveById(activityId);
        
        String pageId = activity.getValue(AbstractPageActivityType.PAGE_ID);
        Page page = _resolver.resolveById(pageId);
        
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, page.getSiteName());
        
        // Retrieve the current context.
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        try
        {
            // Force rendering context.FRONT to resolve URI
            _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
        
            // Recipients
            Map<CMSTag, Set<UserIdentity>> subscribersByTag = _getSubscribersByTag(page, activity);
            for (CMSTag tag : subscribersByTag.keySet())
            {
                // Subject
                String subject = _getMailSubject(page, tag, activity);
                
                // Body
                String mailBodyHTML = _getMailHtmlBody(page, tag, activity);
                
                // Recipients
                List<String> recipients = subscribersByTag.get(tag)
                        .stream()
                        .map(u -> _userManager.getUser(u))
                        .filter(Objects::nonNull)
                        .map(User::getEmail)
                        .filter(StringUtils::isNotBlank)
                        .toList();
                try
                {
                    SendMailHelper.newMail()
                            .withRecipients(recipients)
                            .withSubject(subject)
                            .withHTMLBody(mailBodyHTML)
                            .withAsync(true)
                            .withInlineCSS(false)
                            .sendMail();
                }
                catch (MessagingException | IOException e)
                {
                    getLogger().warn("Could not send a notification e-mail to {}", recipients, e);
                }
            }
        }
        finally 
        {
            // Restore current context
            _renderingContextHandler.setRenderingContext(currentContext);
        }
    }
    
    /**
     * Get all subscribers by tags of the activity
     * @param page the page
     * @param activity the activity
     * @return the map of tags with its subscribers
     */
    protected Map<CMSTag, Set<UserIdentity>> _getSubscribersByTag(Page page, Activity activity)
    {
        Map<CMSTag, Set<UserIdentity>> subscribersByTag = new HashMap<>();
        UserIdentity author = activity.getAuthor();
        
        for (String tagName : activity.getValue(PageUpdatedActivityType.CONTENT_TAGS, new String[0]))
        {
            TagSubscriptionContext context = TagSubscriptionContext.newInstance()
                    .withTag(tagName);
            
            // Get all subscribers to a tag  (resolve groups)
            List<UserIdentity> allSubscribers = _tagSubscriptionType.getSubscribers(page.getSite(), Frequency.INSTANT, BroadcastChannel.MAIL, context)
                .stream()
                .filter(u -> !u.equals(author)) // ignore if it's the author of the activity
                .filter(u -> _rightManager.hasReadAccess(u, page)) // check user read access
                .toList();
            
            // FIXME Very low performance !! all users of a groups will retrieved and for each of them a JCR request will be done
            
            for (UserIdentity subscriber : allSubscribers)
            {
                // For each subscriber, get the computed subscriptions (taking into account user groups)
                // For example, if USER subscribed to TAG but he is part of GROUP subscribed (force) to TAG, #getUserSubscriptions will return only the group subscription
                for (Subscription subscription : _tagSubscriptionType.getUserSubscriptions(page.getSite(), Frequency.INSTANT, BroadcastChannel.MAIL, subscriber, false, context))
                {
                    CMSTag tag = _tagSubscriptionType.getTarget(subscription);
                    Set<UserIdentity> subscribers = subscribersByTag.getOrDefault(tag, new HashSet<>());
                    subscribers.add(subscriber);
                    subscribersByTag.put(tag, subscribers);
                }
            }
        }
        
        return subscribersByTag;
    }
    
    /**
     * Get the mail subject
     * @param page the page
     * @param tag the tag
     * @param activity the activity
     * @return the mail subject
     */
    protected String _getMailSubject(Page page, CMSTag tag, Activity activity)
    {
        Map<String, I18nizableTextParameter> params = Map.of(
            "tag", tag.getTitle(),
            "site", new I18nizableText(page.getSite().getTitle())
        );
        
        I18nizableText i18nSubject = new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_SUBJECT", params);
        return _i18nUtils.translate(i18nSubject, page.getSitemapName());
    }

    /**
     * Get the mail html body
     * @param page the page
     * @param tag the tag
     * @param activity the activity
     * @return the mail html body
     * @throws IOException if an error occurred
     */
    protected String _getMailHtmlBody(Page page, CMSTag tag, Activity activity) throws IOException
    {
        Site site = page.getSite();

        String contentId = activity.getValue(PageUpdatedActivityType.CONTENT_ID);
        Content content = _resolver.resolveById(contentId);
        String lang = content.getLanguage();

        MailBodyBuilder bodyBuilder = TagSubscriptionSummaryMailBodyHelper.newHTMLBody()
                .withLanguage(lang)
                .withTitle(_getMailBodyTitle(tag))
                .withHint(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_MESSAGE_INSTANT"))
                .withContents(Set.of(content))
                .withLink(site.getUrl(), new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_NOTIFICATIONS_MAIL_SITE_LINK"));
            
        Optional<String> pageTagUrl = _pageDAO.findPagedIdsByTag(site.getName(), lang, AbstractSendTagNotificationSummarySchedulable.SUBSCRIPTIONS_CONFIG_PAGE_TAG)
                .stream()
                .map(id -> ResolveURIComponent.resolve("page", id, false, true))
                .findFirst();
        if (pageTagUrl.isPresent())
        {
            bodyBuilder.withFooterLink(pageTagUrl.get(), new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_PREFERENCE_LINK"), "core-ui", "img/mail/icon-letter.png");
        }
        
        return bodyBuilder.build();
    }
    
    private I18nizableText _getMailBodyTitle(CMSTag tag)
    {
        Map<String, I18nizableTextParameter> params = Map.of("tag", tag.getTitle());
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_TITLE", params);
    }
}
