/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.DayOfWeek;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ElementDefinition;

/**
 * Helper for frequency
 */
public class FrequencyHelper implements Component, Serviceable, Initializable, LogEnabled
{
    /** The i18n utils */
    protected static I18nUtils _i18nUtils;
    
    private static long _frequencyDay;
    private static String _frequencyTime;
    
    private static Logger _logger;
    
    /**
     * The frequency type.
     */
    public enum Frequency
    {
        /** Now */
        INSTANT,
        /** Every day */
        DAILY,
        /** Every week */
        WEEKLY,
        /** Every month */
        MONTHLY;
    }
    
    public void enableLogging(Logger logger)
    {
        _logger = logger;        
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _frequencyDay = Config.getInstance().getValue("page-subscription.frequency.day", true, 1L);
        _frequencyTime = Config.getInstance().getValue("page-subscription.frequency.hour", true, "10:00");
    }
    
    /**
     * Get the default day of frequency (weekly or monthly)
     * @return the default day of frequency
     */
    public static long getDefaultFrequencyDay()
    {
        return _frequencyDay;
    }
    
    /**
     * Get the default time of frequency (daily, weekly or monthly)
     * @return the default time of frequency
     */
    public static String getDefaultFrequencyTime()
    {
        return _frequencyTime;
    }
    
    private static I18nizableText _getEntry(long day)
    {
        try
        {
            ElementDefinition def = (ElementDefinition) Config.getModel().getChild("page-subscription.frequency.day");
            return def.getEnumerator().getEntry(day);
        }
        catch (Exception e)
        {
            _logger.error("An error occurred getting day from value '" + day + "'", e);
        }
        
        return null;
    }
    
    /**
     * SAX frequencies
     * @param contentHandler the content handler to sax intos
     * @throws SAXException if an error occured while saxing
     */
    public static void saxFrequencies(ContentHandler contentHandler) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "frequencies");
        for (Frequency frequency : Frequency.values())
        {
            if (frequency != Frequency.INSTANT)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("name", frequency.name());
                XMLUtils.startElement(contentHandler, "frequency", attrs);
                getLabel(frequency).toSAX(contentHandler, "label");
                getSmartLabel(frequency).toSAX(contentHandler, "smartLabel");
                XMLUtils.endElement(contentHandler, "frequency");
            }
        }
        XMLUtils.endElement(contentHandler, "frequencies");
    }
    /**
     * Get the label for the given frequency
     * @param frequency the frequency
     * @return the label
     */
    public static I18nizableText getLabel(Frequency frequency)
    {
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_FREQUENCY_LABEL_" + frequency.name());
    }
    
    /**
     * Get the smart label with day and hour of the given frequency
     * @param frequency the frequency
     * @return the smart label
     */
    public static I18nizableText getSmartLabel(Frequency frequency)
    {
        return getSmartLabel(frequency, null);
    }
    
    /**
     * Get the smart label with day and hour of the given frequency
     * @param frequency the frequency
     * @param timing the frequency timing. Can be null to get the default config param
     * @return the smart label
     */
    public static I18nizableText getSmartLabel(Frequency frequency, FrequencyTiming timing)
    {
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        I18nizableText hourI18n = timing != null && timing.time() != null
                ? new I18nizableText(_getHour(timing.time()))
                : new I18nizableText(_getHour(_frequencyTime));
        switch (frequency)
        {
            case DAILY: 
                params.put("hour", hourI18n);
                break;
            case WEEKLY: 
            case MONTHLY:
                I18nizableText dayI18n  = timing != null && timing.day() != null
                    ? _getEntry(timing.day())
                    : _getEntry(_frequencyDay);
                if (dayI18n != null)
                {
                    params.put("day", dayI18n);
                }
                params.put("hour", hourI18n);
                break;
            default: // Do nothing ..
        }
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_FREQUENCY_SMART_LABEL_" + frequency.name(), params);
    }
    
    /**
     * Get full label with frequency and broadcast channel
     * @param frequency the frequency
     * @param channels the broadcast channels
     * @return the frequency and broadcast channel label
     */
    public static I18nizableText getFullLabel(Frequency frequency, List<BroadcastChannel> channels)
    {
        return getFullLabel(frequency, channels, null);
    }
    
    /**
     * Get full label with frequency and broadcast channel
     * @param frequency the frequency
     * @param channels the broadcast channels
     * @param timing the frequency timing of the full label. Can be null to get the default config param
     * @return the frequency and broadcast channel label
     */
    public static I18nizableText getFullLabel(Frequency frequency, List<BroadcastChannel> channels, FrequencyTiming timing)
    {
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        I18nizableText dayI18n  = timing != null && timing.day() != null
                ? _getEntry(timing.day())
                : _getEntry(_frequencyDay);
        if (dayI18n != null)
        {
            params.put("day", dayI18n);
        }
        I18nizableText broadcastChannelKey = new I18nizableText("plugin.page-subscription", _getBroadcastChannelKey(channels));
        params.put("broadcastChannel", broadcastChannelKey);
        
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_FREQUENCY_" + frequency.name() + "_AND_BROADCAST_CHANNEL_LABEL", params);
    }
    
    private static String _getBroadcastChannelKey(List<BroadcastChannel> channels)
    {
        if (channels.contains(BroadcastChannel.MAIL) && channels.contains(BroadcastChannel.SITE))
        {
            return "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_BROADCAST_CHANNEL_MSG_SITE_AND_MAIL";
        }
        else if (channels.contains(BroadcastChannel.SITE))
        {
            return "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_BROADCAST_CHANNEL_MSG_SITE";
        }
        else if (channels.contains(BroadcastChannel.MAIL))
        {
            return "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_BROADCAST_CHANNEL_MSG_MAIL";
        }
        
        return "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_BROADCAST_CHANNEL_MSG_NONE";
    }
    
    private static String _getHour(String time)
    {
        String stringFormat = _i18nUtils.translate(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_FREQUENCY_HOUR_FORMAT"));
        
        SimpleDateFormat defaultFormat = new SimpleDateFormat("HH:mm");
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(stringFormat);
        try
        {
            return simpleDateFormat.format(defaultFormat.parse(time));
        }
        catch (ParseException e)
        {
            return time;
        }
    }
    
    /**
     * Get the frequency timing of the subscription
     * @param subscription the subscription
     * @return the frequency timing
     */
    public static FrequencyTiming getTiming(Subscription subscription)
    {
        FrequencyTiming forceFrequencyTiming = subscription.getForceFrequencyTiming();
        long day = forceFrequencyTiming.day() != null ? forceFrequencyTiming.day() : getDefaultFrequencyDay();
        String time = StringUtils.isNotBlank(forceFrequencyTiming.time()) ? forceFrequencyTiming.time() : getDefaultFrequencyTime();
        
        return new FrequencyTiming(day, time);
    }
    
    /**
     * Get cron from frequency and timing
     * @param frequency the frequency
     * @param frequencyTiming the timing
     * @return the cron expression
     */
    public static String getCron(Frequency frequency, FrequencyTiming frequencyTiming)
    {
        int minutes = (int) frequencyTiming.minutes();
        int hour = (int) frequencyTiming.hour();
        
        switch (frequency)
        {
            case DAILY: 
            {
                CronScheduleBuilder builder = CronScheduleBuilder.dailyAtHourAndMinute(hour, minutes);
                CronTrigger trigger = (CronTrigger) builder.build();
                return trigger.getCronExpression();
            }
            case WEEKLY: 
            {
                CronScheduleBuilder builder = CronScheduleBuilder.weeklyOnDayAndHourAndMinute(_getDay(frequencyTiming), hour, minutes);
                CronTrigger trigger = (CronTrigger) builder.build();
                return trigger.getCronExpression();
            }
            case MONTHLY: 
            {
                return "0 " + minutes + " " + hour + " ? * " + _getDay(frequencyTiming) + "#1";
            }
            default:
                throw new IllegalArgumentException("Can create subscription schedulable with the given frequency: " + frequency);
        }
    }
    
    private static int _getDay(FrequencyTiming frequencyTiming)
    {
        // Mapping Ametys day API to Quartz day API
        // Monday 1 => 2
        // Tuesday 2 => 3
        // Wednesday 3 => 4
        // Thursday 4 => 5
        // Friday 5 => 6
        // Saturday 6 => 7
        // Sunday 7 => 1
        int day = frequencyTiming.day().intValue();
        return day == 7
            ? 1
            : day + 1;
    }
    
    /**
     * Get the date expressions for a given frequency to get activities before the notification date
     * @param frequency the frequency
     * @param notificationDate the notification date
     * @return date expressions
     */
    public static List<Expression> getDateExpressions(Frequency frequency, ZonedDateTime notificationDate)
    {
        List<Expression> dateExprs = new ArrayList<>();
        switch (frequency)
        {
            case INSTANT: 
            {
                // For this frequency, get all news since the last past 6 months
                // FIXME ca sort d'où ces 6 mois ??
                ZonedDateTime startDate = notificationDate.minusMonths(6); 
                dateExprs.add(new DateExpression("date", Operator.GT, DateUtils.asDate(startDate)));
                break;
            }
            case DAILY: 
            {
                // Frequency is each day at HH:mm
                // => get contents updated 
                // - between yesterday-1 at HH:mm and yesterday at HH:mm if current time is before HH:mm 
                // - or between yesterday at HH:mm and today at HH:mm if current time is after HH:mm 
                ZonedDateTime startDate = notificationDate.minusDays(1);
                
                dateExprs.add(new DateExpression("date", Operator.GT, DateUtils.asDate(startDate)));
                dateExprs.add(new DateExpression("date", Operator.LT, DateUtils.asDate(notificationDate)));
                break;
            }
            case WEEKLY: 
            {
                // Frequency is each D day the week at HH:mm
                // => get contents updated 
                // - between D-14 at HH:mm and D-7 at HH:mm if current time is before D day at HH:mm 
                // - or between D-7 day at HH:mm and D day at HH:mm if current time is after D day at HH:mm 
                ZonedDateTime startDate = notificationDate.minusDays(7);

                dateExprs.add(new DateExpression("date", Operator.GT, DateUtils.asDate(startDate)));
                dateExprs.add(new DateExpression("date", Operator.LT, DateUtils.asDate(notificationDate)));
                break;
            }
            case MONTHLY: 
            {
                // Frequency is each 1st D day the month M at HH:mm
                // => get contents updated 
                // - between 1st D day of the month M-2 at HH:mm and 1st D day of the month M-1 if current time is before 1st D day of the month at HH:mm 
                // - or between 1st D day of the month M-1 at HH:mm and 1st D day of the month M at HH:mm if current time is after 1st D day of the month at HH:mm 
                ZonedDateTime startDate = notificationDate.minusMonths(1);
            
                dateExprs.add(new DateExpression("date", Operator.GT, DateUtils.asDate(startDate)));
                dateExprs.add(new DateExpression("date", Operator.LT, DateUtils.asDate(notificationDate)));
                break;
            }
            default:
        }
        
        return dateExprs;
    }
    
    /**
     * Get the notification date for a given frequency and timing
     * @param frequency the frequency
     * @param timing the timing
     * @return the notification date
     */
    public static ZonedDateTime getNotificationDate(Frequency frequency, FrequencyTiming timing)
    {
        ZonedDateTime now = ZonedDateTime.now();
        
        long day = timing.day();
        int hour = (int) timing.hour();
        int minute = (int) timing.minutes();
        
        switch (frequency)
        {
            case INSTANT: 
            {
                // Return the today date
                return now.truncatedTo(ChronoUnit.DAYS);
            }
            case DAILY:
            {
                // Frequency is each day at HH:mm
                // => get notification date
                // - notification date is yesterday at HH:mm if current time is before HH:mm 
                // - or notification date is today at HH:mm if current time is after HH:mm 
                return _isCurrentDayBeforeDailyTime(now, hour, minute)
                        ? now.minusDays(1).truncatedTo(ChronoUnit.DAYS).withHour(hour).withMinute(minute)
                        : now.truncatedTo(ChronoUnit.DAYS).withHour(hour).withMinute(minute);
            }
            case WEEKLY:
            {
                // Frequency is each D day the week at HH:mm
                // => get notification date
                // - notification date is D-7 at HH:mm if current time is before D day at HH:mm 
                // - or notification date is D day at HH:mm if current time is after D day at HH:mm 
                ZonedDateTime dayOfWeek = now.with(TemporalAdjusters.previousOrSame(DayOfWeek.of((int) day))); // get previous or same day of the week
                return dayOfWeek.getDayOfMonth() == now.getDayOfMonth() && _isCurrentDayBeforeDailyTime(now, hour, minute)
                        ? dayOfWeek.minusDays(7).truncatedTo(ChronoUnit.DAYS).withHour(hour).withMinute(minute)
                        : dayOfWeek.truncatedTo(ChronoUnit.DAYS).withHour(hour).withMinute(minute);
            }
            case MONTHLY:
            {
                // Frequency is each 1st D day the month M at HH:mm
                // => get notification date
                // - notification date is the 1st D day of the month M-1 if current time is before 1st D day of the month at HH:mm 
                // - or notification date is the 1st D day of the month M at HH:mm if current time is after 1st D day of the month at HH:mm 
                ZonedDateTime dayOfMonth = now.with(TemporalAdjusters.firstInMonth(DayOfWeek.of((int) day))); // get first day of the month
                return now.getDayOfMonth() < dayOfMonth.getDayOfMonth() || now.getDayOfMonth() == day && _isCurrentDayBeforeDailyTime(now, hour, minute)
                        ? dayOfMonth.minusMonths(1).truncatedTo(ChronoUnit.DAYS).withHour(hour).withMinute(minute)
                        : dayOfMonth.truncatedTo(ChronoUnit.DAYS).withHour(hour).withMinute(minute);
            }
            default:
        }
        
        return null;
    }
    
    private static boolean _isCurrentDayBeforeDailyTime(ZonedDateTime now, int hour, int minute)
    {
        return now.getHour() < hour // is before if the current hour is before the given time hour
            || now.getHour() == hour && now.getMinute() < minute; // or is before if the current hour is equal to the given time hour and the current minute is before the given time minute
    }
}
