/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.dao;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.context.PageSubscriptionContext;
import org.ametys.plugins.pagesubscription.type.PageSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.rights.PageRightAssignmentContext;

/**
 * DAO for page subscriptions
 */
public class PageSubscriptionsDAO extends AbstractSubscriptionsDAO
{
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The page subscription type */
    protected PageSubscriptionType _pageSubscriptionType;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _pageSubscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);
    }
    
    /**
     * Get subscriptions of a page
     * @param pageId the page id
     * @return the results
     */
    @Callable(rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = PageRightAssignmentContext.ID)
    public Map<String, Object> getPageSubscriptions(String pageId)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        Page page = _resolver.resolveById(pageId);
        
        return _getPageSubscriptions(page, user);
    }
    
    private Map<String, Object> _getPageSubscriptions(Page page, UserIdentity user)
    {
        Map<String, Object> results = new HashMap<>();
        
        Set<UserIdentity> subscribers = getPageSubscribers(page);
        
        results.put("hasSubscribed", subscribers.stream()
                .filter(u -> u.equals(user))
                .findAny()
                .isPresent()
        );
        results.put("nbSubscribers", subscribers.size());
        
        return results;
    }
    
    /**
     * Subscribe the current user to a page 
     * @param pageId the page id
     * @return the results
     */
    @Callable(rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = PageRightAssignmentContext.ID)
    public Map<String, Object> subscribePage(String pageId)
    {
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);

        Map<String, Object> results = new HashMap<>();
        
        UserIdentity user = _currentUserProvider.getUser();
        Page page = _resolver.resolveById(pageId);
        
        try
        {
            // Force default the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            Pair<Frequency, List<BroadcastChannel>> defaultPreference = _getDefaultPreference(page.getSite());
            PageSubscriptionContext context = PageSubscriptionContext.newInstance()
                    .withPage(page);
            _pageSubscriptionType.subscribe(page.getSite(), user, defaultPreference.getLeft(), defaultPreference.getRight(), context);
            results.put("success", true);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred subscribing to the page with id '{}'", pageId, e);
            results.put("success", false);
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
        
        results.putAll(_getPageSubscriptions(page, user));
        return results;
    }
    
    private Pair<Frequency, List<BroadcastChannel>> _getDefaultPreference(Site site)
    {
        String frequencyValue = site.getValue("page-default-frequency", true, Frequency.DAILY.name());
        return "NOMAIL".equals(frequencyValue)
            ? Pair.of(Frequency.INSTANT, List.of(BroadcastChannel.SITE))
            : Pair.of(Frequency.valueOf(frequencyValue), List.of(BroadcastChannel.MAIL, BroadcastChannel.SITE));
    }
    
    /**
     * Unsubscribe the current user to a page 
     * @param pageId the page id
     * @return the results
     */
    @Callable(rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = PageRightAssignmentContext.ID)
    public Map<String, Object> unsubscribePage(String pageId)
    {
        Map<String, Object> results = new HashMap<>();
        
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        UserIdentity user = _currentUserProvider.getUser();
        Page page = _resolver.resolveById(pageId);
        
        try
        {
            // Force default the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            PageSubscriptionContext context = PageSubscriptionContext.newInstance()
                    .withPage(page);
            _pageSubscriptionType.unsubscribe(page.getSite(), user, context);
            results.put("success", true);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred unsubscribing to the page with id '{}'", pageId, e);
            results.put("success", false);
            return results;
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
        
        results.putAll(_getPageSubscriptions(page, user));
        return results;
    }
    
    /**
     * Get the page subscribers
     * @param page the page
     * @return the subscribers
     */
    public Set<UserIdentity> getPageSubscribers(Page page)
    {
        PageSubscriptionContext context = PageSubscriptionContext.newInstance()
                .withPage(page);
        return _pageSubscriptionType.getSubscribers(page.getSite(), null, null, context);
    }
    
}
