/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.dao;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.SubscriptionException;
import org.ametys.plugins.pagesubscription.SubscriptionException.Type;
import org.ametys.plugins.pagesubscription.type.SubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract DAO for subscriptions
 */
public abstract class AbstractSubscriptionsDAO extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** the Avalon context */
    protected Context _context;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    private void _checkSubscription(Subscription subscription)
    {
        UserIdentity user = _currentUserProvider.getUser();
        boolean isSubscriber = subscription.getSubscriber()
            .filter(u -> u.equals(user))
            .isPresent();
        if (!isSubscriber)
        {
            throw new SubscriptionException("Can access to the subscription with id '" + subscription.getId() + "' because the subscriber is not the current user.", Type.NO_ACCESS);
        }
    }
    
    /**
     * Get the current user subscription from the given id
     * @param subscriptionId the subscription id
     * @return the subscription as JSON
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getSubscription(String subscriptionId)
    {
        return getSubscription(subscriptionId, true);
    }
    
    /**
     * Get the subscription from the given id
     * @param subscriptionId the subscription id
     * @param checkCurrentUser <code>true</code> to check if the subscription belong to the current user
     * @return the subscription as JSON
     */
    public Map<String, Object> getSubscription(String subscriptionId, boolean checkCurrentUser)
    {
        try
        {
            Subscription subscription = _resolver.resolveById(subscriptionId);
            if (checkCurrentUser)
            {
                _checkSubscription(subscription);
            }
            
            return Map.of("success", true, "subscription", subscription.getSubscriptionType().subscriptionToJSON(subscription));
        }
        catch (Exception e)
        {
            getLogger().error("Fail to get subscription with id '{}'", subscriptionId, e);
            return Map.of("success", false);
        }
    }
    
    /**
     * Edit the current user subscription with the given id
     * @param subscriptionId the subscription id
     * @param frequency the frequency 
     * @param broadcastChannels the broadcast channels
     * @return the subscription as JSON
     */ 
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editSubscription(String subscriptionId, String frequency, List<String> broadcastChannels)
    {
        return editSubscription(subscriptionId, frequency, null, broadcastChannels, true);
    }
    
    /**
     * Edit the subscription with the given id
     * @param subscriptionId the subscription id
     * @param frequency the frequency 
     * @param forcedTiming the force frequency timing. Can be null if not forced
     * @param broadcastChannels the broadcast channels
     * @param checkCurrentUser <code>true</code> to check if the subscription belong to the current user
     * @return a Map with success status and the subscription as JSON
     */ 
    @SuppressWarnings("unchecked")
    public Map<String, Object> editSubscription(String subscriptionId, String frequency, FrequencyTiming forcedTiming, List<String> broadcastChannels, boolean checkCurrentUser)
    {
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            Subscription subscription = _resolver.resolveById(subscriptionId);
            if (checkCurrentUser)
            {
                _checkSubscription(subscription);
            }
            
            SubscriptionType subscriptionType = subscription.getSubscriptionType();
            
            Frequency subFrequency = StringUtils.isNotBlank(frequency)
                    ? Frequency.valueOf(frequency)
                    : subscription.getFrequency();
            
            List<BroadcastChannel> subChannels = broadcastChannels != null
                    ? broadcastChannels.stream()
                            .map(BroadcastChannel::valueOf)
                            .toList()
                    : subscription.getBroadcastChannels();
            
            if (forcedTiming != null)
            {
                subscriptionType.editForceSubscription(subscription, subFrequency, subChannels, forcedTiming, null);
            }
            else
            {
                subscriptionType.editSubscription(subscription, subFrequency, subChannels, null);
            }
            
            return Map.of("success", true, "subscription", subscriptionType.subscriptionToJSON(subscription));
        }
        catch (Exception e)
        {
            getLogger().error("Unable to edit subscription with id '{}'", subscriptionId, e);
            return Map.of("success", false);
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Remove the current user subscription with the given id
     * @param subscriptionId the subscription id
     * @return true if successfully subscribe, false otherwise
     * @throws Exception if an error occurred
     */ 
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public boolean unsubscribe(String subscriptionId) throws Exception
    {
        return unsubscribe(subscriptionId, true);
    }
    
    /**
     * Remove the subscription with the given id
     * @param subscriptionId the subscription id
     * @return true if successfully subscribe, false otherwise
     * @param checkCurrentUser <code>true</code> to check if the subscription belong to the current user
     * @throws Exception if an error occurred
     */ 
    public boolean unsubscribe(String subscriptionId, boolean checkCurrentUser) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            Subscription subscription = _resolver.resolveById(subscriptionId);
            if (checkCurrentUser)
            {
                _checkSubscription(subscription);
            }
            
            subscription.getSubscriptionType().unsubscribe(subscription);
            return true;
        }
        catch (Exception e)
        {
            getLogger().error("Fail to unsubscribe to suscription with id {}", subscriptionId, e);
            return false;
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
}
