/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ObservationConstants;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.type.SubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Observer to unsubscribe a user or group on deletion
 */
public class RemoveSubscriptionOnSubscriberDeletedObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    /** The site manager */
    protected SiteManager _siteManager;
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;

    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
    }
    
    public boolean supports(Event event)
    {
        String eventId = event.getId();
        return ObservationConstants.EVENT_USER_DELETED.equals(eventId)
            || ObservationConstants.EVENT_GROUP_DELETED.equals(eventId);
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        boolean isUserEvent = switch (event.getId())
        {
            case ObservationConstants.EVENT_USER_DELETED -> true;
            case ObservationConstants.EVENT_GROUP_DELETED -> false;
            default ->
            throw new IllegalArgumentException("Unexpected event id: " + event.getId());
        };
        
        // one of the two is null but that allow using the same for loops for both case
        // without fetching the subscriber every iteration
        UserIdentity user = (UserIdentity) event.getArguments().get(ObservationConstants.ARGS_USER);
        GroupIdentity group = (GroupIdentity) event.getArguments().get(ObservationConstants.ARGS_GROUP);
        
        try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
        {
            for (Site site : sites)
            {
                for (String typeId : _subscriptionTypeEP.getExtensionsIds())
                {
                    SubscriptionType subscriptionType = _subscriptionTypeEP.getExtension(typeId);
                    if (isUserEvent)
                    {
                        subscriptionType.unsubscribeAll(site, user);
                    }
                    else
                    {
                        subscriptionType.unsubscribeAll(site, group);
                    }
                }
            }
        }
    }

    public int getPriority()
    {
        return Integer.MAX_VALUE;
    }
}
