/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserManager;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for broadcast channel
 */
public class BroadcastChannelHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The current user provider */
    protected static CurrentUserProvider _currentUserProvider;
    
    /** The user manager */
    protected static UserManager _userManager;
    
    /**
     * The broadcast channel of the subscription
     */
    public enum BroadcastChannel
    {
        /** The mail broadcast channel */
        MAIL,
        /** The site broadcast channel */
        SITE;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
    }
    
    /**
     * Get the label of the broadcast channel
     * @param broadcastChannel the broadcast channel
     * @return the broadcast channel label
     */
    public static I18nizableText getLabel(BroadcastChannel broadcastChannel)
    {
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_BROADCAST_CHANNEL_LABEL_" + broadcastChannel.name());
    }
    
    /**
     * Get the smart label of the broadcast channel
     * @param broadcastChannel the broadcast channel
     * @return the broadcast channel smart label
     */
    public static I18nizableText getSmartLabel(BroadcastChannel broadcastChannel)
    {
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        switch (broadcastChannel)
        {
            case MAIL: 
                String email = _getCurrentUserMail();
                params.put("mail", new I18nizableText(email));
                break;
            default: // Do nothing ..
                break;
        }
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_BROADCAST_CHANNEL_SMART_LABEL_" + broadcastChannel.name(), params);
    }
    
    private static String _getCurrentUserMail()
    {
        return Optional.ofNullable(_currentUserProvider.getUser())
            .map(_userManager::getUser)
            .filter(Objects::nonNull)
            .map(User::getEmail)
            .orElse(StringUtils.EMPTY);
    }
}
