/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.TagSubscriptionContext;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * DAO for tag subscriptions
 */
public class TagSubscriptionsDAO extends AbstractSubscriptionsDAO
{
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The tag provider extension point */
    protected TagProviderExtensionPoint _tagProviderEP;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) smanager.lookup(TagProviderExtensionPoint.ROLE);
    }
    
    /**
     * Get the tag subscriptions for the current user as JSON
     * @param siteName the site name
     * @param rootTagName the root tag name
     * @return the subscriptions as JSON
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getTagSubscriptions(String siteName, String rootTagName)
    {
        List<Subscription> userSubscriptions = _getTagSubscriptions(siteName, rootTagName);
        
        List<Map> userSubscriptions2Json = userSubscriptions.stream()
            .map(s -> s.getSubscriptionType().subscriptionToJSON(s))
            .toList();
        
        boolean hasAvailableTags = !getAvailableTags(userSubscriptions, siteName, rootTagName).isEmpty();
        
        return Map.of(
            "tagSubscriptions", userSubscriptions2Json,
            "hasAvailableTags", hasAvailableTags
        );
    }
    
    private List<Subscription> _getTagSubscriptions(String siteName, String rootTagName)
    {
        TagSubscriptionType subscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        
        Site site = _siteManager.getSite(siteName);
        return subscriptionType.getUserSubscriptions(site, _currentUserProvider.getUser(), false, TagSubscriptionContext.newInstance().withRootTag(rootTagName).withSiteName(siteName));
    }
    
    /**
     * Get the available tags for current user
     * @param siteName the site name
     * @param rootTagName the root tag name
     * @return the list of available tags
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, Object>> getAvailableTags(String siteName, String rootTagName)
    {
        List<Subscription> subscriptions = _getTagSubscriptions(siteName, rootTagName);
        
        return getAvailableTags(subscriptions, siteName, rootTagName)
                .stream()
                .map(t -> Map.of(
                    "label", t.getTitle(),
                    "name", t.getName()
                ))
                .toList();
    }
    
    /**
     * Get the available tags 
     * @param subscriptions the susbcriptions
     * @param siteName the site name
     * @param rootTagName the root tag name
     * @return the list of available tags
     */
    public Set<CMSTag> getAvailableTags(List<Subscription> subscriptions, String siteName, String rootTagName)
    {
        List<String> tagNames = subscriptions.stream()
            .map(s -> s.<String>getValue(TagSubscriptionType.TAG))
            .toList();
        
        CMSTag tag = _tagProviderEP.getTag(rootTagName, Map.of("siteName", siteName));
        return Optional.ofNullable(tag)
            .map(CMSTag::getTags)
            .map(Map::values)
            .orElseGet(() -> List.of())
            .stream()
            .filter(t -> !tagNames.contains(t.getName()))
            .collect(Collectors.toSet());
    }
    
    /**
     * Add a tag subscription
     * @param siteName the site name holding the subscription
     * @param rootTagName the root tag name
     * @param tag the tag
     * @param frequency the frequency
     * @param broadcastChannels the broadcast channels
     * @return the tag subscriptions as JSON
     * @throws Exception if an error occurred
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> addTagSubscription(String siteName, String rootTagName, String tag, String frequency, List<String> broadcastChannels) throws Exception
    {
        List<Map<String, Object>> fieldsInError = _getFieldsInError(tag, frequency, broadcastChannels);
        if (!fieldsInError.isEmpty())
        {
            return Map.of("success", false, "fieldsInError", fieldsInError);
        }
        
        Request request = ContextHelper.getRequest(_context);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            Site site = _siteManager.getSite(siteName);
            TagSubscriptionType tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);

            UserIdentity user = _currentUserProvider.getUser();
            TagSubscriptionContext context = TagSubscriptionContext.newInstance()
                .withTag(tag);
            
            List<BroadcastChannel> broadcastChannelsAsEnum = broadcastChannels.stream()
                .map(c -> BroadcastChannel.valueOf(c))
                .toList();
            
            Subscription subscription = tagSubscriptionType.subscribe(site, user, Frequency.valueOf(frequency), broadcastChannelsAsEnum, context);
            
            return Map.of("success", true, "subscription", tagSubscriptionType.subscriptionToJSON(subscription), "hasAvailableTags", !getAvailableTags(siteName, rootTagName).isEmpty());
        }
        catch (Exception e)
        {
            getLogger().error("Unable to add tag subscription for site name '{}' and tag '{}'", siteName, tag, e);
            return Map.of("success", false);
        }
        finally 
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    private List<Map<String, Object>> _getFieldsInError(String tag, String frequency, List<String> broadcastChannels)
    {
        List<Map<String, Object>> fieldsInError = new ArrayList<>();
        if (StringUtils.isBlank(tag))
        {
            fieldsInError.add(Map.of(
                    "fieldName", "tag",
                    "error", new I18nizableText("plugin.page-subscription", "TAG_SUBSCRIPTION_HELPER_FORM_TAG_MANDATORY_ERROR")));
        }
        
        if (StringUtils.isBlank(frequency))
        {
            fieldsInError.add(Map.of(
                    "fieldName", "frequency",
                    "error", new I18nizableText("plugin.page-subscription", "TAG_SUBSCRIPTION_HELPER_FORM_FREQUENCY_MANDATORY_ERROR")));
        }
        
        if (broadcastChannels == null || broadcastChannels.isEmpty())
        {
            fieldsInError.add(Map.of(
                    "fieldName", "broadcastChannel",
                    "error", new I18nizableText("plugin.page-subscription", "TAG_SUBSCRIPTION_HELPER_FORM_CHANNEL_MANDATORY_ERROR")));
        }
        
        return fieldsInError;
    }
}
