/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.pagesubscription.notification.PageNotificationsHelper;
import org.ametys.plugins.pagesubscription.notification.TagNotificationsHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ObservationConstants;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.activities.AbstractPageActivityType;
import org.ametys.web.activities.PageUpdatedActivityType;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;

/**
 * Observer to clear notifications cache
 */
public class ClearNotificationsCacheOnPageActivityObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The tag notifications helper */
    protected TagNotificationsHelper _tagNotificationsHelper;

    /** The page notifications helper */
    protected PageNotificationsHelper _pageNotificationsHelper;

    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _tagNotificationsHelper = (TagNotificationsHelper) manager.lookup(TagNotificationsHelper.ROLE);
        _pageNotificationsHelper = (PageNotificationsHelper) manager.lookup(PageNotificationsHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        if (ObservationConstants.EVENT_ACTIVITY_CREATED.equals(event.getId()))
        {
            Activity activity = (Activity) event.getArguments().get(ObservationConstants.ARGS_ACTIVITY);
            return activity != null && activity.getActivityType() instanceof AbstractPageActivityType;
        }
        
        return false;
    }

    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Activity activity = (Activity) event.getArguments().get(ObservationConstants.ARGS_ACTIVITY);
        
        String pageId = activity.getValue(AbstractPageActivityType.PAGE_ID);
        Page page = _resolver.resolveById(pageId);
        _pageNotificationsHelper.clearCache(page.getSiteName(), page.getId());
        
        if (activity.getActivityType() instanceof PageUpdatedActivityType)
        {
            String contentId = activity.getValue(PageUpdatedActivityType.CONTENT_ID);
            Content content = _resolver.resolveById(contentId);
            String siteName = content instanceof WebContent webContent ? webContent.getSiteName() : null;
            // Clear cache on each site (site = null) if the content is not a web content
            for (String tagName : content.getTags())
            {
                _tagNotificationsHelper.clearCache(siteName, tagName);
            }
        }
    }
}
