/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.schedulable.tag;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Runnable to send mail to group tag subscriptions
 */
public class SendGroupTagNotificationSummaryRunnable extends DefaultRunnable
{
    /**
     * The send group tag notitification summary runnable
     * @param subscription the subscription
     * @param id the id
     * @param label the label
     * @param description the description
     */
    public SendGroupTagNotificationSummaryRunnable(Subscription subscription, String id, I18nizableText label, I18nizableText description)
    {
        super(
            id, 
            label, 
            description, 
            FireProcess.CRON, 
            FrequencyHelper.getCron(subscription.getFrequency(), _getTiming(subscription)), 
            _getSchedulableId(subscription), 
            false, // No removable
            false, // No modifiable
            false, // No desactivable
            null, 
            true, 
            UserPopulationDAO.SYSTEM_USER_IDENTITY, 
            Map.of(
                "subscriptionId", subscription.getId()
            )
        );
    }
    
    private static String _getSchedulableId(Subscription subscription)
    {
        return subscription.getFrequency().name().toLowerCase() + "-group-tag-schedulable";
    }
    
    private static FrequencyTiming _getTiming(Subscription subscription)
    {
        FrequencyTiming forceFrequencyTiming = subscription.getForceFrequencyTiming();
        long day = forceFrequencyTiming.day() != null ? forceFrequencyTiming.day() : FrequencyHelper.getDefaultFrequencyDay();
        String time = StringUtils.isNotBlank(forceFrequencyTiming.time()) ? forceFrequencyTiming.time() : FrequencyHelper.getDefaultFrequencyTime();
        
        return new FrequencyTiming(day, time);
    }
}
