/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
AmetysFront.UserTagSubscriptions = {
    
    TAG_SUBSCRIPTIONS_ATTR: "data-ametys-tag-subscriptions",
    
    TAG_SUBSCRIPTION_ID_ATTR: "data-ametys-subscription-id",
    
    TAG_SUBSCRIPTIONS_RESULTS_SELECTOR: '[data-ametys-tag-subscription-role="results"]',
    
    TAG_SUBSCRIPTIONS_NORESULT_SELECTOR: '[data-ametys-tag-subscription-role="no-result"]',
    
    TAG_SUBSCRIPTIONS_STATUS_SELECTOR: '[data-ametys-tag-subscription-role="status"]',
    
    TAG_SUBSCRIPTIONS_ADD_BUTTON_SELECTOR: '[data-ametys-tag-subscription-role="add"]',
    
    TAG_SUBSCRIPTIONS_FREQUENCY_SELECTOR: '[data-ametys-tag-subscription-role="frequency"]',
    
    TAG_SUBSCRIPTIONS_SMART_FREQUENCY_SELECTOR: '[data-ametys-tag-subscription-role="smart-frequency"]',
    
    TAG_SUBSCRIPTIONS_FULL_FREQUENCY_SELECTOR: '[data-ametys-tag-subscription-role="full-frequency"]',
    
    /** The tag subscription template selector */
    TAG_SUBSCRIPTION_TEMPLATE_SELECTOR : "tag-subscription-template",
    
    /** The tag subscription template selector */
    FORCE_TAG_SUBSCRIPTION_TEMPLATE_SELECTOR : "forced-tag-subscription-template",
    
    /** Regexp for subscription' data such as %%title%% */
    SUBSCRIPTION_VALUE_REGEXP: /%%(\S+?)%%/,
    
    /** The selector to tag subscriptions root */
    SUBSCRIPTION_BY_TAG_ROOT_SELECTOR: "data-ametys-subscription-by-tag",
    
    /**
     * Load subscriptions
     * @param {String} uniqueId the uniqueId id
     * @param {String} rootTag the roog tag name for themes
     * @param {Function} [callback] the callback function to invoked after loading subscription
     */
    loadSubscriptions: function (uniqueId, rootTag, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.TagSubscriptionsDAO',
            methodName: 'getTagSubscriptions',
            parameters: [AmetysFront.getAppParameters().siteName, rootTag],
            callback: {
                handler: this._loadSubscriptionsCb,
                scope: this,
                arguments: {
                    uniqueId: uniqueId,
                    callback: callback
                }
            }
        });
    },
    
    /**
     * @private
     * Callback after loading subscription
     * @param {Object} response the response with all subscriptions
     * @param {Object} args the arguments
     * @private
     */
    _loadSubscriptionsCb: function(response, args)
    {
        let $wrapper = $j('[' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ATTR + '="' + args.uniqueId + '"]');
        let $results = $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_RESULTS_SELECTOR);
        
        let uniqueId = args.uniqueId;
        let subscriptions = response["tagSubscriptions"] || [];
        
        this._fillSubscriptions(uniqueId, $results, subscriptions);
        
        if (subscriptions.length == 0)
        {
            $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_NORESULT_SELECTOR).show();
        }
        
        if (response.hasAvailableTags)
        {
            $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ADD_BUTTON_SELECTOR).show();
        }
        else
        {
            $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ADD_BUTTON_SELECTOR).hide();
        }
        
        if (typeof(args.callback) == 'function')
        {
            args.callback(response, args);
        }
    },
    
    /**
     * Insert subscription' items
     * @param {String} uniqueId the unique identifier
     * @param {HTMLElement} $results The results element to insert items
     * @param {Object[]} subscriptions The list of subscriptions
     * @private
     */
    _fillSubscriptions: function(uniqueId, $results, subscriptions)
    {
        if (subscriptions)
        {
            $results.empty();
            
            let $subscriptionTpl = $j('#' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTION_TEMPLATE_SELECTOR + '-' + uniqueId);
            let $forceSubscriptionTpl = $j('#' + AmetysFront.UserTagSubscriptions.FORCE_TAG_SUBSCRIPTION_TEMPLATE_SELECTOR + '-' + uniqueId);
            
            for (let subscription of subscriptions)
            {
                // Insert subscription item
                let htmlTpl = subscription.isForced ? $forceSubscriptionTpl.clone().html() : $subscriptionTpl.clone().html();
                let htmlSubscription = this._fillTemplate(htmlTpl, subscription);
                $results.append(htmlSubscription);
            }
        }
    },
    
    /**
     * Fill a susbcription's item from HTML templete with susbcription data
     * @param {String} templateHtml The HTML of the template
     * @param {Object} data the susbcription data as JSON object
     * @return the HTML object
     * @private
     */
    _fillTemplate: function (templateHtml, data)
    {
        while ((result =  AmetysFront.UserTagSubscriptions.SUBSCRIPTION_VALUE_REGEXP.exec(templateHtml)) !== null) 
        {
            // replace string matched
            let value = this._getValue(data, result[1]);
            if (Array.isArray(value) || value !== null && typeof value === 'object')
            {
                value = JSON.stringify(value).replace(/"/g, '&quot;');
            }
            templateHtml = templateHtml.replace(result[0], value);
        }
        
        return templateHtml;
    },
    
    /**
     * Get value of json object for the given data name
     * @param {Object} json the object as JSON
     * @param {String} dataName the data name
     * @return the value
     * @private
     */
    _getValue: function (json, dataName)
    {
        // dateName format can be : name, path.to.data
        let paths = dataName.split('.');
        
        var data = json;
        for (var i=0; i < paths.length; i++)
        {
            var name = paths[i];
            if (i == paths.length -1)
            {
                return data[name] || '';
            }
            
            if (data[name])
            {
                data = data[name];
            }
            else
            {
                return '';
            }
        }
        
        return '';
    },
    
    /**
     * Get subscription data
     * @param {String} susbriptionId the id of subscription
     * @param {Function} the function to invoke after retrieving subscription's data
     */
    getSubscription: function(subscriptionId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.TagSubscriptionsDAO',
            methodName: 'getSubscription',
            parameters: [subscriptionId],
            callback: {
                handler: function(response) {
                    if (response.success && response.subscription)
                    {
                        callback(response.subscription);
                    }
                    else
                    {
                        AmetysFront.Utils.error("{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_GET_SUBSCRIPTION_ERROR}}", "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_ERROR_TITLE}}");
                    }
                },
                scope: this
            }
        });
    },
    
    /**
     * Get available tags
     * @param {String} rootTagName the root tag name for themes
     * @param {Function} the function to invoke after retrieving subscription's data
     */
    getAvailableTags: function(rootTagName, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.TagSubscriptionsDAO',
            methodName: 'getAvailableTags',
            parameters: [AmetysFront.getAppParameters().siteName, rootTagName],
            callback: {
                handler: function(tags) {
                    callback(tags || []);
                },
                scope: this
            }
        });
    },
    
    /**
     * Initialize subscription form for adding a new subscription
     * @param {String} rootTagName the root tag name for themes
     * @param {HTMLElement} form the form with subscription inputs
     * @param {Function} [callback] the callback function to invoked after getting available tags
     */
    addSubscription: function(rootTag, form, callback)
    {
        AmetysFront.UserTagSubscriptions.getAvailableTags(rootTag, function(tags) {
            let $form = $j(form);
            $form[0].reset();
            $form.find('input[name="subscriptionId"]').val('');
            
            let $select = $form.find('select[name="tag"]');
            $select.empty();
            
            // Add placeholder
	        var placeholderOpt = new Option("{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_SELECT_EMPTY_OPTION}}", "");
	        $j(placeholderOpt).attr("disabled", true);
	        $j(placeholderOpt).attr("selected", true);
	        $j(placeholderOpt).attr("hidden", true);
	        $select.append($j(placeholderOpt));
	        
	        for (var tag of tags)
	        {
	            var opt = new Option(tag.label, tag.name);
	            $select.append($j(opt));
	        }
            
            if (typeof(callback) == 'function')
            {
                callback(form, false);
            }
        });
    },
    
    /**
     * Initialize subscription form for editing a subscription
     * @param {String} subscriptionId the subscription id
     * @param {HTMLElement} form the form with subscription inputs
     * @param {Function} [callback] the callback function to invoked after getting subscription
     * @private
     */
    editSubscription: function (subscriptionId, form, callback)
    {
        AmetysFront.UserTagSubscriptions.getSubscription(subscriptionId, function(subscription) {
            let $form = $j(form);
            $form[0].reset();
            
            $form.find('input[name="subscriptionId"]').val(subscriptionId);
            
            let $select = $form.find('select[name="tag"]');
            $select.empty();
            
            var opt = new Option(subscription.tag.label, subscription.tag.name);
	        $j(opt).attr("selected", true);
	        $select.append($j(opt));
	        $select.disableSelection();
            $select.attr("readonly", "readonly");
            
            $form.find("*[name='frequency'][value='" + subscription.frequency.name + "']").prop("checked", true);
            
            for (var broadcastChannel of subscription.broadcastChannel)
	        {
	            $form.find("*[name='broadcastChannel'][value='" + broadcastChannel.name + "']").prop("checked", true);
	        }
            
            if (typeof(callback) == 'function')
	        {
	            callback(form, true);
	        }
        })
    },
    
    /**
     * Save subscription
     * @param {String} uniqueId The unique id
     * @param {String} rootTagName The root tag name
     * @param {HTMLElement} form the form with subscription inputs
     * @param {Function} successcallback the callback function to invoke in case of success
     * @param {Function} errorCallback the callback function to invoke in case of error
     */
    saveSubscription: function (uniqueId, form, rootTagName, successcallback, errorCallback)
    {
        let $form = $j(form);
        
        var values = {};
		$j.each($form.serializeArray(), function(i, field) {
            if (values.hasOwnProperty(field.name))
            {
                var val = values[field.name];
                if (!Array.isArray(val))
                {
                    val = [val];
                }
                val.push(field.value);
                values[field.name] = val;
            }
            else
            {
                values[field.name] = field.value;
            }
		});
        
        // Handle errors
        var errors = [];
        if (!values.tag)
        {
            errors.push({fieldName: 'tag', error: "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_FORM_TAG_MANDATORY_ERROR}}"});
        }
        if (!values.frequency)
        {
            errors.push({fieldName: 'frequency', error: "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_FORM_FREQUENCY_MANDATORY_ERROR}}"});
        }
        if (!values.broadcastChannel)
        {
            errors.push({fieldName: 'broadcastChannel', error: "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_FORM_CHANNEL_MANDATORY_ERROR}}"});
        }
        
        values.broadcastChannel = values.broadcastChannel || [];
        
        if (errors.length)
        {
            errorCallback(form, {"fieldsInError": errors});
            return;
        }
        
        let editMode = values.subscriptionId != null && values.subscriptionId != '';
        
        let params = [];
        if (editMode)
        {
	        params = [values.subscriptionId, values.frequency, Array.isArray(values.broadcastChannel) ? values.broadcastChannel : [values.broadcastChannel]]
        }
        else
        {
            params = [AmetysFront.getAppParameters().siteName, rootTagName, values.tag, values.frequency, Array.isArray(values.broadcastChannel) ? values.broadcastChannel : [values.broadcastChannel]]
        }
        
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.pagesubscription.dao.TagSubscriptionsDAO',
            methodName: editMode ? 'editSubscription' : 'addTagSubscription',
            parameters: params,
            callback: {
                handler: this._saveSubscriptionCb,
                scope: this,
                arguments: {
                    uniqueId: uniqueId,
                    callback: successcallback,
                    errorCallback: errorCallback,
                    subscriptionId: values.subscriptionId,
                    editMode: editMode,
                    form: form
                }
            }
        });
    },
    
    /**
     * Callback function after saving a susbcription
     * @param {Object} response the server response
     * @param {Object} args the arguments
     */
    _saveSubscriptionCb: function (response, args)
    {
        var editMode = args.editMode;
        if (response.success)
        {
            let $wrapper = $j('[' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ATTR + '="' + args.uniqueId + '"]');
            
            let data = response.subscription;
            if (editMode)
            {
                // Update subscription
                let $subcriptionEl = $j('[' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTION_ID_ATTR + '="' + args.subscriptionId + '"]');
                $subcriptionEl.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_FREQUENCY_SELECTOR).html(data.frequency.label);
                $subcriptionEl.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_SMART_FREQUENCY_SELECTOR).html(data.frequency.smartLabel);
                $subcriptionEl.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_FULL_FREQUENCY_SELECTOR).html(data.frequency.fullLabel);
                
                $subcriptionEl.attr("tabindex", -1);
                $subcriptionEl.trigger('focus');
                
                $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_STATUS_SELECTOR).html("{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_EDIT_SUBSCRIPTION_SUCCESS}}")
            }
            else
            {
                // Insert new subscription
                let $results = $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_RESULTS_SELECTOR);
                let $subscriptionTpl = $j('#' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTION_TEMPLATE_SELECTOR + '-' + args.uniqueId);
                
                let htmlTpl = $subscriptionTpl.clone().html();
                let htmlSubscription = this._fillTemplate(htmlTpl, data);
                $results.append(htmlSubscription);
                
                let $subcriptionEl = $j('[' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTION_ID_ATTR + '="' + data.id + '"]');
                $subcriptionEl.attr("tabindex", -1);
                $subcriptionEl.trigger('focus');
                
                if (!response.hasAvailableTags)
                {
                    // No more available tags, hide button
                    $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ADD_BUTTON_SELECTOR).hide();
                }
            }
            
            if (typeof(args.callback) == 'function')
            {
                args.callback(data, args);
            }
        }
        else
        {
            if (!response.fieldsInError)
            {
	            var errorMsg = args.editMode ? "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_EDIT_SUBSCRIPTION_ERROR}}" : "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_SUBSCRIBE_ERROR}}";
	            AmetysFront.Utils.error(errorMsg, "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_ERROR_TITLE}}");
            }
            
            if (typeof(args.errorCallback) == 'function')
            {
                args.errorCallback(args.form, response);
            }
        }
    },
    
    /**
     * Unsubscribe
     * @param {String} uniqueId the uniqueId id
     * @param {String} subscriptionId the subscription id
     * @param {Function} [callback] the callback function to invoked after removing subscription
     * @param {Boolean} [skipConfirm] true to skip confirm before deleting
     */
    unsubscribe: function (uniqueId, subscriptionId, callback, skipConfirm)
    {
        let me = this;
        
        function __internalUnsubscribe()
        {
            AmetysFront.ServerComm.callMethod({
                role: 'org.ametys.plugins.pagesubscription.dao.TagSubscriptionsDAO',
                methodName: 'unsubscribe',
                parameters: [subscriptionId],
                callback: {
                    handler: me._unsubscribeCb,
                    scope: me,
                    arguments: {
                        uniqueId: uniqueId,
                        subscriptionId: subscriptionId,
                        callback: callback
                    }
                },
                errorMessage: true
            });
        }
        
        if (skipConfirm)
        {
            __internalUnsubscribe()
        }
        else
        {
            AmetysFront.Utils.confirm("{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_UNSUBSCRIBE_CONFIRM_TITLE}}", "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_UNSUBSCRIBE_CONFIRM_MSG}}", function () {
                __internalUnsubscribe()
            });
        }
    },
    
    /**
     * Callback function invoked after unsubscribing
     * @param {Boolean} success true if unsubscribe successfully
     * @param {Object[]} args the additional arguments
     * @private
     */
    _unsubscribeCb: function(success, args)
    {
        if (success)
        {
            // Remove subscription item
            let $subcriptionEl = $j('[' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTION_ID_ATTR + '="' + args.subscriptionId + '"]');
            $subcriptionEl.remove();

            let $wrapper = $j('[' + AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ATTR + '="' + args.uniqueId + '"]');
            let $result = $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_RESULTS_SELECTOR);
            let count = $result.children().length;
            if (count == 0)
            {
                $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_NORESULT_SELECTOR).show();
            }
            
            $result.attr("tabindex", -1);
            $result.trigger('focus');
            
            // New tag available => show add button
            $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_ADD_BUTTON_SELECTOR).show();
            $wrapper.find(AmetysFront.UserTagSubscriptions.TAG_SUBSCRIPTIONS_STATUS_SELECTOR).html("{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_UNSUBSCRIBE_SUCCESS}}")
            
            if (typeof(args.callback) == 'function')
            {
                args.callback(args.subscriptionId);
            }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_UNSUBSCRIBE_ERROR}}", "{{i18n plugin.page-subscription:TAG_SUBSCRIPTION_HELPER_ERROR_TITLE}}")
        }
    }
}
