/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.plugins.pagesubscription.type.SubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Remove subscription from unknown user
 */
public class SubscriptionDataPolicy extends AbstractLogEnabled implements PersonalDataPolicy, Serviceable
{
    /** The site manager */
    protected SiteManager _siteManager;
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    private Period _retentionPeriod;

    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        Long config = Config.getInstance().<Long>getValue("page-subscription.retention", false, null);
        // disable the processing if config is negative
        _retentionPeriod = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    
    public AnonymizationResult process(UserStatusInfo userStatusInfo)
    {
        if (_retentionPeriod == null)
        {
            return AnonymizationResult.TOO_EARLY;
        }
        else if (userStatusInfo.getMissingSinceDate().isBefore(ZonedDateTime.now().minus(_retentionPeriod)))
        {
            UserIdentity userIdentity = userStatusInfo.getUserIdentity();
            try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
            {
                long deletedSubscriptions = 0;
                for (Site site : sites)
                {
                    for (String typeId : _subscriptionTypeEP.getExtensionsIds())
                    {
                        SubscriptionType subscriptionType = _subscriptionTypeEP.getExtension(typeId);
                        deletedSubscriptions += subscriptionType.unsubscribeAll(site, userIdentity);
                    }
                }
                
                if (deletedSubscriptions > 0)
                {
                    ForensicLogger.info("data.policy.gdpr.remove.page.subscriptions", Map.of("handled", Long.toString(deletedSubscriptions), "identity", userIdentity), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                    return AnonymizationResult.PROCESSED;
                }
                else
                {
                    return AnonymizationResult.NO_DATA;
                }
            }
            catch (Exception e)
            {
                getLogger().error("An error prevented to remove all the subscription of the unknown user '" + userIdentity, e);
                return AnonymizationResult.ERROR;
            }
        }
        else
        {
            return AnonymizationResult.TOO_EARLY;
        }
    }
}
